package player.rawmaterialsolver;

import java.util.Comparator;

import edu.neu.ccs.csu670.components.cspsolver.CspSolverState;
import edu.neu.ccs.demeterf.demfgen.lib.List;
import edu.neu.ccs.demeterf.demfgen.lib.Map;
import edu.neu.ccs.evergreen.ir.Relation;
import edu.neu.ccs.evergreen.ir.RelationI;
import edu.neu.ccs.satsolver.OutputI;
import edu.neu.ccs.satsolver.SATSolverUtil;
import gen.Assignment;
import gen.Constraint;
import gen.Literal;
import gen.RawMaterial;
import gen.RelationNr;
import gen.Sign;
import gen.Variable;

/**
 * An efficient implementation of {@link CspSolverState} for 
 * {@link RawMaterial}s
 */
public class RawMaterialSolverState implements CspSolverState {

  private final List<Literal> assigned;
  private final Map<Variable, List<Integer>> variables;
  private final Map<Integer, Constraint> constraints;
  private final double sat;
  private final double unsat;
  private final double total;
  private final OutputI lookAhead;

  public RawMaterialSolverState(RawMaterial rm) {
    Map<Variable, List<Integer>> variables = Map.create(new Comparator<Variable>() {
      public int compare(Variable a, Variable b) {
        return a.v.compareTo(b.v);
      }
    });
    Map<Integer, Constraint> constraints = Map.create();
    double sat = 0;
    double unsat = 0;
    double total = 0;
    
    int key = 0;
    InputInitial inputInitial = new InputInitial();
    
    
    for (Constraint c : rm.instance.cs) {
      inputInitial.addConstraint(c.r, c.w);
      
      constraints = constraints.put(key, c);
      
      total += c.w.v;
      if (c.r.v == 255) {
        sat += c.w.v;
      } else if (c.r.v == 0) {
        unsat += c.w.v;
      } else {
        RelationI relation = new Relation(3, c.r.v);
        int pos = 0;
        for (Variable v : c.vs) {
          if (!relation.isIrrelevant(pos)) {
            if (variables.containsKey(v)) {
              variables = variables.remap(v, variables.get(v).push(key));
            } else {
              variables = variables.put(v, List.<Integer>create(key));
            }
          }
          ++pos;
        }
      }
      
      ++key;
    }
    
    this.assigned = List.create();
    this.variables = variables;
    this.constraints = constraints;
    this.sat = sat;
    this.unsat = unsat;
    this.total = total;
    this.lookAhead = SATSolverUtil.calculateBias(inputInitial);
  }
  
  private RawMaterialSolverState(
      Variable v, 
      Sign s, 
      RawMaterialSolverState prev) {
    Map<Integer, Constraint> constraints = prev.constraints;
    double sat = prev.sat;
    double unsat = prev.unsat;
    double total = prev.total;
    
    InputUpdate inputUpdate = new InputUpdate(
        total, 
        prev.lookAhead.getPolynomial());
    
    for (Integer key : prev.variables.get(v)) {
      if (constraints.containsKey(key)) {
        Constraint c = constraints.get(key);
        RelationI relation = new Relation(3, c.r.v);
     
        int n = relation.reduce(c.vs.index(v), s.sign());
        RelationNr newRelationNr = new RelationNr(n);
        inputUpdate.reduceConstraint(c.r, newRelationNr, c.w);
        
        if (n == 255) {
          sat += c.w.v;
          constraints = constraints.remove(key);
        } else if (n == 0) {
          unsat += c.w.v;
          constraints = constraints.remove(key);
        } else {
          constraints = constraints.remap(
              key, 
              new Constraint(c.w, newRelationNr, c.vs)); 
        }
      }
    }
    
    this.assigned = prev.assigned.push(new Literal(s, v));
    this.variables = prev.variables.remove(v);
    this.constraints = constraints;
    this.sat = sat;
    this.unsat = unsat;
    this.total = total;
    this.lookAhead = SATSolverUtil.updateBias(inputUpdate);    
  }
  
  public CspSolverState setVariable(Variable v, Sign sign) {
    return new RawMaterialSolverState(v, sign, this);
  }

  public double fractionSatisfied() {
    return sat/total;
  }
  
  public double fractionUnsatisfied() {
    return unsat/total;
  }
  
  public Assignment getAssignment() {
    return new Assignment(assigned);
  }
  
  public Iterable<Variable> getUnassignedVariables() {
    return variables.keys();
  }
  
  public OutputI getLookAhead() {
    return lookAhead;
  }
}
