package com.ibm.lab.usingjdt.threadeddebugging;

/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;

/**
 * Class <code>NumberGenerator</code> is a superclass for
 * <code>PrimeNumberGenerator</code> and <code>PrimeFactorialGenerator </code>.
 * It is used in a number of the exercises in Chapters 29 and 30 to illustrate
 * different capabilities of the Java Development Tools (JDT).
 * 
 * This version of <code>NumberGenerator</code> has been modified slightly to
 * make it multithreaded. One of the modifications is to make <code>NumberGenerator</code>
 * a subclass of <code>Thread</code>.
 */
public class NumberGenerator extends Thread {

	/** 
	 * Field <code>quantity</code> is the number of numbers to generate, starting
	 * with 2. 
	 */	
	protected int quantity;
	
	/**
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		generateNumbers();
		outputResults(getName());
	}

	/**
	 * Method <code>outputResults </code> outputs the generated numbers.
	 * @param prefix String added to the front of each line of output. Used
	 * to distinguish output from different threads.
	 */
	public void outputResults(String threadName) {
		Date d = new Date();
		String prefix = "";
		if (threadName != "") {
			prefix = threadName + ": ";
		}
		System.out.println(prefix + d.toString());
		for (int i = 0; i < getResults().length; i++) {
			System.out.println(prefix + getResults()[i]);
		}
	}

	/**
	 * Method <code>sortResults</code> sorts the numbers in reverse order.
	 * The example may seem a bit contrived. It is included to demonstrate generating
	 * inner classes.
	 */
	public void sortResults() {
		Arrays.sort(getResults(),
		new Comparator() {
			public int compare(Object o1, Object o2) {
				if (((Integer) o1).intValue() < ((Integer) o2).intValue()) {
					return 1;
				} else {
					return -1;
				}
			}

		});
	}

	/**
	 * Method <code>generateNumbers</code> is overridden by subclasses of 
	 * <code>NumberGenerator</code> to generate specific types of numbers.
	 */
	public void generateNumbers() {
	}

	/**
	 * Method <code>getResults</code> returns the generated numbers.
	 * @return Object[]
	 */
	public Object[] getResults() {
		return results;
	}

	/**
	 * Method <code>setResults</code> sets generated prime numbers.
	 * @param results Array of generated prime numbers
	 */
	public void setResults(Object[] results) {
		this.results = results;
	}

	/**
	 * Method <code>setQuantity</code> sets the number of numbers to generate.
	 * @param quan Number of numbers to be generated
	 */
	public void setQuantity(int quan) {
		this.quantity = quan;
	}

	/**
	 * Method <code>getQuantity</code> returns the number of numbers to generate.
	 * @return int
	 */
	public int getQuantity() {
		return quantity;
	}

	/**
	 * Field <code>results</code> contains the generated numbers.
	 */
	protected Object[] results;
}
