package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceProxy;
import org.eclipse.core.resources.IResourceProxyVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IProgressMonitor;

import java.util.Map;

import com.ibm.lab.soln.resources.IResourceIDs;

/**
 * Builder for .readme files.  Creates and HTML stub file for each readme file in 
 * a project's readme folder to demonstrate the structure of a builder and how an 
 * <code>IResourceDelta</code> can be processed to find changes of interest to 
 * a builder.
 * <p>
 * The builder will proecess files in the current project, and register its 
 * intent to process files in any referenced project.  This means that even 
 * though the builder is not associated to a project, the fact that a project 
 * with this builder has a project reference means that the target project
 * will also be processed.
 * <p>
 * Source includes trace logic that can be enabled or disabled based on the 
 * setting of the <code>traceEnabled</code> field. The trace logic uses 
 * <code>System.out.println()</code> statements to create trace entries.
 * If you want to visualize the flow of build processing, 
 * uncomment these statements before starting a test cycle.
 * </p>
 * @see org.eclipse.core.resources.IncrementalProjectBuilder
 * @see org.eclipse.core.resources.IResourceDelta
 */
public class ReadmeBuilder extends IncrementalProjectBuilder {
  // Counter for trace messages to track the build events
  private int counter = 0;
  // switch to control write of trace data
  private boolean traceEnabled = false;
  // Values used to prefix trace messages
  private String buildMode = "<build>";
  private String buildMark = " ----> ";

  /**
   * The required public no-argument constructor.
   */
  public ReadmeBuilder() {
  }

  /**
   * Finds builder parameters defined in the plugin.xml.
   * Parameters are printed to the console if the <code>System.out.println()</code> 
   * statements are uncommented.
   * 
   * @see org.eclipse.core.runtime.IExecutableExtension#setInitializationData(IConfigurationElement, String, Object)
   */
  public void setInitializationData(
    IConfigurationElement config,
    String propertyName,
    Object data)
    throws CoreException {
    super.setInitializationData(config, propertyName, data);

    IConfigurationElement[] run = config.getChildren();
    IConfigurationElement[] parms = run[0].getChildren();
    for (int i = 0; i < parms.length; i++) {
      traceMsg(
        "Passed Parameters: "
          + parms[i].getAttribute("name")
          + " = "
          + parms[i].getAttribute("value"));

    }

  }

  /**
   * This method allows a builder to get ready. Called early in the build cycle.
   * 
   * @see import org.eclipse.core.resources.IncrementalProjectBuilder#startupOnInitialize()
   */
  protected void startupOnInitialize() {
    traceMsg("Readme Builder Initialize - startupOnInitialize()");
  }

  /**
   * The build method implementation required for any incremental builder. 
   * This demonstration of build processing reacts to a <code>FULL_BUILD</code>,
   * <code>AUTO_BUILD</code>, or <code>INCREMENTAL_BUILD</code> build request.
   * The processing logic to "build" the .readme files is kept in the
   * <code>ReadmeVisitor</code> class and is used for all build types.
   */
  protected IProject[] build(int kind, Map args, IProgressMonitor monitor)
    throws CoreException {

    counter++;
    traceMsg("In Readme Builder - build()");
    traceMsg("Build for project: " + getProject());

    //* Optional method to trace kind of build to console
    //      checkBuildType(kind);

    if (kind == IncrementalProjectBuilder.FULL_BUILD) {
      // Full build
      IResourceDelta delta = getDelta(getProject());
      traceMsg("Build Type: Full Build");
      processFull(getProject());

      if (watchReferecedProjects() != null)
        processFull4ReferencedProjects(watchReferecedProjects());

    } else { // build with a delta (Auto/Incremental)

      IResourceDelta delta = getDelta(getProject());
      traceMsg("Process delta for base project");
      if (delta != null)
        processDelta(delta);
      // Hit when project created
      else
        traceMsg("Build Type: Delta/no-delta Build");
      if (watchReferecedProjects() != null)
        processDelta4ReferencedProjects(watchReferecedProjects());

    }

    traceMsg("<-- End Readme Builder");
    return watchReferecedProjects();
  }

  /**
   * Prints builder kind trace messages to the console 
   */
public void checkBuildType(int kind) {
    switch (kind) {

      case IncrementalProjectBuilder.INCREMENTAL_BUILD :
        traceMsg("Build Type: Incremental Build");
        break;

      case IncrementalProjectBuilder.AUTO_BUILD :
        traceMsg("Build Type: Auto Build");
        break;

      case IncrementalProjectBuilder.FULL_BUILD :
        traceMsg("Build Type: Full Build");
    }
  }

  /**
   * Returns an array of projects that are of interest for the next build.
   * If this array is returned by the build method, the build can get a delta 
   * for these projects during the next build cycle.
   * 
   * The projects returned are those referenced by the current project, if none
   * are referenced a null is returned.
   */
  public  IProject[] watchReferecedProjects() {

    IProject[] refedProjects;
    try {
      refedProjects = getProject().getReferencedProjects();
    } catch (CoreException e) {
      refedProjects = null;
    }

    if (refedProjects.length != 0) {
      traceMsg(">-->Found Referenced Projects" + refedProjects);

      return refedProjects;

    } else {
      traceMsg(">-->No referenced projects found");
      return null;
    }

  }

  /**
   * Performs a <code>FULL_BUILD</code> by visiting all nodes in the resource
   * tree under readme folder for the specified project.  The readme folder must
   * exist. 
   * The <code>ReadmeVisitor</code> class is used to process the .readme files 
   * when found in the project tree.
   * @param iProject
   */
  public void processFull(IProject iProject) {
    // focus on readme folder
    final IFolder readmeFolder = iProject.getFolder(IResourceIDs.README_FOLDER);

    // if there is a readme folder in the project

    if (readmeFolder.exists()) {

      // Use logic in readme visitor used to process a delta
      final ReadmeVisitor visitLogic = new ReadmeVisitor();

      // Create resource visitor logic
      IResourceVisitor myVisitor = new IResourceVisitor() {
        public boolean visit(IResource resource) throws CoreException {
          traceMsg("Full Build visit(" + resource + ")");
          if (resource.getType() == IResource.FILE) {
            if (resource.getName().endsWith(".readme")) {
              visitLogic.processReadme(resource, readmeFolder);
            }
          }

          return true; // carry on
        }
      };
      
	  // Create resource proxy visitor logic @Since 2.1
	  IResourceProxyVisitor myProxyVisitor = new IResourceProxyVisitor() {
		public boolean visit(IResourceProxy resourceProxy) throws CoreException {
		  traceMsg("Full Build visit(" + resourceProxy.getName() + ") - a proxy");
		  if (resourceProxy.getType() == IResource.FILE) {
			if (resourceProxy.getName().endsWith(".readme")) {
			  visitLogic.processReadme(resourceProxy.requestResource(), (IContainer)readmeFolder);
			}
		  }

		  return true; // carry on
		}
	  };

      // Process the readme folder using the resource visitor just created
      // Can use the resource visitor or resource proxy visitor
      // Only one of the folder.accept methods below should be used (pick one)
      try {
		// readmeFolder.accept(myVisitor);
        readmeFolder.accept(myProxyVisitor, IResource.NONE);
      } catch (CoreException e) {
        e.printStackTrace();
      }
    }
  }

  /**
   * Performs a full buils on any referenced projects.
   */
  public void processFull4ReferencedProjects(IProject[] refedProjects) {

    for (int i = 0; i < refedProjects.length; i++) {
      IProject iProject = refedProjects[i];
      processFull(iProject);
    }
  }

  /**
   * The <code>IResourceDelta</code> passed is processed by a 
   * <code>IResourceDeltaVisitor</code> when the delta contains information
   * about changes in the readme folder.  
   * A null delta exists when the project is created.
   * <p>
   * Multiple visit invocation statements are included for the different 
   * resource visitors provided in this resource programming demonstration package.
   * You may wish to invoke other visitors to further explore the <code>IResourceDelta</code>.
   * <p>
   * 
   * @see org.eclipse.core.resources.IResourceDelta
   */
  public void processDelta(IResourceDelta delta) throws CoreException {

    // Use defined path to get portion of resource delta that is of interest
    IResourceDelta focusDelta =
      delta.findMember(IResourceIDs.README_FOLDER);

    if (focusDelta == null) {
      traceMsg(
        "Build Ignored: Delta does not contain "
          + IResourceIDs.README_FOLDER
          + " changes.");
    } else {
      // Visitor does the work
      traceMsg(
        "Build includes a "
          + IResourceIDs.README_FOLDER
          + " folder Delta: Time to visit...");

      // Processes delta by looking for readme files
      // Creates an html stub when one is found
      focusDelta.accept(new ReadmeVisitor());

      //* If you just want to trace information about the resource delta
      //* the change the visitor used to visit the ResourceDelta.

      //delta.accept(new ResourceDeltaPrinter("build" + 1));

    }
  }

  /**
   * Attempts to process readme changes in another project where the 
   * builder is not registered. Other projects are those referenced by
   * the current project.
   * @throws CoreException
   */
  public void processDelta4ReferencedProjects(IProject[] refedProjects)
    throws CoreException {

    for (int i = 0; i < refedProjects.length; i++) {
      IProject iProject = refedProjects[i];

      IResourceDelta refDelta = getDelta(iProject);
      traceMsg("Process delta for project: " + iProject);
      if (refDelta != null)
        processDelta(refDelta);
    }

  }

  /**
  * Write trace statements.  
  * System.out.println with prefix tagging used for simplicity.
  */
  private void traceMsg(String msg) {
    if (IResourceIDs.TRACE_ENABLED | traceEnabled)
      System.out.println(
        buildMode
          + "<"
          + getProject()
          + "> "
          + counter
          + "\t\t\t"
          + buildMark
          + msg);
  }
}