package com.ibm.lab.soln.resources.markers;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;

import com.ibm.lab.soln.resources.EDUResourcesPlugin;
import com.ibm.lab.soln.resources.IResourceIDs;

/**
 * Implements support for a set of revolving markers that identify the most 
 * recently edited files. A fixed maximum number of markers are created.
 * Internally a <code>HashMap</code> is used to track the marker-resource
 * relationship and a <code>Vector</code> to implement a FIFO list of 
 * recently edited files and backtrack to the marker when one is removed.
 * 
 */
public class RecentEdits {
  // track resources so we don't have multiple markers per resource
  private HashMap markerMap = null;
  // track resources to enforce upper limit on number of markers added
  private Vector limitList = null;
  // Limit for number of markers to be kept
  private int limit;
  // prefix for markers - sorted by name in task list, by resource in bookmakers
  private String prefix = "aRE: ";

  /**
   * Constructor for <code>RecentEdits</code> which loads all markers into 
   * the internal <code>HashMap</code> and <code>Vector</code>. This class is
   * instantiated by the <code>RecentEditsRCL</code> resource change listener.
   */
  public RecentEdits() {
    super();
    limit = getLimit();
    limitList = new Vector();
    markerMap = new HashMap();

    IMarker[] markers = null;
    try {
      markers =
        EDUResourcesPlugin.getWorkspace().getRoot().findMarkers(
          IResourceIDs.MARKER_ID,
          false,
          IResource.DEPTH_INFINITE);
    } catch (CoreException e) {
      e.printStackTrace();
    }

    for (int i = 0; i < markers.length; i++) {
      if (i < limit) {
        markerMap.put(markers[i].getResource(), markers[i]);
        limitList.add(markers[i].getResource());
      } else {
        try {
          markers[i].delete();
        } catch (CoreException e) {
          e.printStackTrace();
        }
      }
    }
  }

  /**
   * Adds a marker to the passed resource and keeps a reference to the 
   * resource in the internal <code>HashMap</code> and <code>Vector</code>.
   * If the limit for the number of markers has been hit, the oldest marker 
   * is removed before a new one is created. Markers are only added if
   * they have not been added before, the resource is not derived, and 
   * the resource name does not start with a "." to indicate a system file.
   * Derived resources are typically created by a builder, and not subject 
   * to direct user editing.
   * <p>
   * The marker is defined as a subtype of bookmark and problemmarker so  
   * the marker will be shown in the Bookmark and Tasks view.
   * </p>
   * @param res  Resource to which a marker will be added.
   */
  public void addReditMarker(IResource res) throws CoreException {
    // do not add marker unless 
    if (recentEditsEnabled(res) // if resource not blocked by property
      & !markerMap.containsKey(res) // not already there 
      & (!res.isDerived()) // not derived
      & (!res.getName().startsWith("."))) // not a system file
      {

      // if beyond limit remove last from vector list and the matching 
      // entry in the map table, then remove the marker
      if (markerMap.size() == limit) {
        IResource oldRes = (IResource)limitList.remove(0);
        IMarker oldMarker = (IMarker)markerMap.get(oldRes);
        markerMap.remove(oldRes);
        oldMarker.delete();
      }

      // add the marker to the resource and then save a marker and resource
      // reference in the internal HashMap and Vector
      IMarker editied_marker = null;
      editied_marker = res.createMarker(IResourceIDs.MARKER_ID);
      editied_marker.setAttribute(IMarker.SEVERITY, 0);
      editied_marker.setAttribute(IMarker.MESSAGE, prefix + res.getName() + " ");

      // @since 2.1
      // Optionally make the REdit marker a transient marker.
      // These are not saved in the workspace.
      if (!getMarkerState())
        editied_marker.setAttribute(IMarker.TRANSIENT, true);

      markerMap.put(res, editied_marker);
      limitList.add(res);
    }
  }

  /**
   * Determines if a marker that is being used to track recently edited
   * files has been deleted, and if so, removes the corresponding entries from 
   * the internal <code>HashMap</code> and <code>Vector</code>.
   * 
   * @param res
   * @param delta
   */
  public void checkReditMarker(IResource res, IResourceDelta delta) {
    // Given the list of changed markers
    IMarkerDelta[] markerdelta = delta.getMarkerDeltas();

    // If the marker was removed and is one of ours
    for (int i = 0; i < markerdelta.length; i++) {
      if (markerdelta[i].getKind() == IResourceDelta.REMOVED)
        // If the removed marker was being tracked
        if (markerMap.containsValue(markerdelta[i].getMarker())) {
          // Remove from vector 
          limitList.remove(res);
          // Find the tracked marker and remove
          Collection col = markerMap.values();
          IMarker holdElement = null;
          for (Iterator iter = col.iterator(); iter.hasNext();) {
            IMarker element = (IMarker)iter.next();
            if (element.getId() == markerdelta[i].getMarker().getId())
              holdElement = element;
          }
          col.remove(holdElement);
        }
    }
  }
  /**
   * Identifies if the resource is ok for recent edit tracking. Property value, 
   * when not defined or set to true, means no tracking.
   * 
   */
  public boolean recentEditsEnabled(IResource resource) {

    try {
      String propValue =
        resource.getPersistentProperty(IResourceIDs.REDIT_PROPERTY_KEY);

      if ((propValue != null) && propValue.equals("True"))
        return false;
      else
        return true;
    } catch (CoreException e) {
      // deal with exception

    }
    return false;

  }

  /**
  * Returns a limit value. Is based on a call to the peference key in the 
  * <code>com.ibm.lab.soln.dialogs</code> plug-in if available, or a hardcoded
  * value of 4.
  * 
  * @return int
  */
  public int getLimit() {
    Plugin softDialogRef = Platform.getPlugin("com.ibm.lab.soln.dialogs");
    if (softDialogRef != null) {
      int limit =
        softDialogRef.getPluginPreferences().getInt(IResourceIDs.REDIT_PREFERENCES_KEY);
      if (limit != 0)
        return limit;
    }
    // yes, we hardcode the default
    return 4;
  }

  /**
  * Returns a boolean value. Is based on a call to the peference key in the 
  * <code>com.ibm.lab.soln.dialogs</code> plug-in if available, or a hardcoded
  * value of true.
  * 
  * @return int
  */
  public boolean getMarkerState() {
    boolean markerState = true;
    Plugin softDialogRef = Platform.getPlugin("com.ibm.lab.soln.dialogs");
    if (softDialogRef != null) {
      markerState =
        softDialogRef.getPluginPreferences().getBoolean(
          IResourceIDs.REDIT_PREFERENCES_KEY2);
    }
    // return the value
    return markerState;
  }

}
