package com.ibm.lab.soln.resources.nature_builder;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */
 
import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Shell;

import com.ibm.lab.soln.resources.EDUResourcesPlugin;
import com.ibm.lab.soln.resources.IResourceIDs;
/**
 * Simple implementation of a nature that customizes a project by adding the 
 * <code>ReadmeBuilder</code> to the list of builders found in a project description.
 */
public class CustomNature implements IProjectNature {
   // To hold associated project reference
   private IProject project;
   // switch to control write of trace data
   private boolean traceEnabled = true;
   /**
    * CustomNature default constructor.
    */
   public CustomNature() {
      super();
   }
   /** 
    * Customizes the project by adding a builder, the ReadmeBuilder in this scenario.
    * 
    * @see org.eclipse.core.resources.IProjectNature#configure()
    */
   public void configure() throws CoreException {

      // Add the ReadmeBuilder to the project
      addReadmeBuilderToBuildSpec(project);

   }
   /**
   * Adds the ReadmeBuilder to the project description for the selected project
   * if it does not already exist.
   *
   * @param project  selected project resource
   */

   public void addReadmeBuilderToBuildSpec(IProject project) throws CoreException {

      // get project description and then the associated build commands 
      IProjectDescription desc = project.getDescription();
      ICommand[] commands = desc.getBuildSpec();

      // determine if builder already associated
      boolean found = false;
      for (int i = 0; i < commands.length; ++i) {
         if (commands[i].getBuilderName().equals(IResourceIDs.BUILDER_ID)) {
            found = true;
            break;
         }
      }

      // add builder if not already in project
      if (!found) {
         ICommand command = desc.newCommand();
         command.setBuilderName(IResourceIDs.BUILDER_ID);
         // create map with arguments specific to builder in project here 
         // command.setArguments(Map args);
         ICommand[] newCommands = new ICommand[commands.length + 1];

         // Add it before other builders. 
         System.arraycopy(commands, 0, newCommands, 1, commands.length);
         newCommands[0] = command;
         desc.setBuildSpec(newCommands);
         project.setDescription(desc, null);
         // Confirm Builder Add
         resultInformation(
            "Soln: Resource - Add ReadmeBuilder Request",
            "ReadmeBuilder was added.");
         //        break;

      } else {

         // Problem with Builder Add
         resultError(
            "Soln: Resource - Add ReadmeBuilder Request",
            "Error adding the ReadmeBuilder to the project.");
      }
   }

   /**
    * Used to show action results.
    * 
    * @see org.eclipse.jface.dialogs.MessageDialog
    */
   protected void resultError(String title, String msg) {
      // Indicate Error
      if (traceEnabled)
         // trace only to console
         System.out.println(title + msg);
      else {
         // user interaction response
         Shell shell = EDUResourcesPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell();
         MessageDialog.openError(shell, title, msg); }

   }



   /**
   * Used to show action results.
   * 
   * @see org.eclipse.jface.dialogs.MessageDialog
   */
   protected void resultInformation(String title, String msg) {
      // Confirm Result

      if (traceEnabled)
         // trace only to console
         System.out.println(title + msg);
      else {
         // user interaction response
         Shell shell = EDUResourcesPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell();
         MessageDialog.openInformation(shell, title, msg); }


   }



   /**
    * Write trace statements.  
    * System.out.println with prefix tagging used for simplicity.
    */
   void traceMsg(String msg) {
      if (traceEnabled)
         System.out.println(msg);
   }

   /** 
    * Removes any nature customization and private resources as may be required.
    * <p>
    * The ReadmeBuilder is not removed here, but is removed automatically 
    * by the platform as the plugin.xml definition links the builder to the nature.
    * </p>
    * <p>
    * This works when the nature identifies the builder using a 
    * <code></code><builder id="a.b.c"/></code> entry and the 
    * builder states that it has a nature using a 
    * <code><builder hasNature="true"> ... </builder></code> entry.
    * </p>
    * @see org.eclipse.core.resources.IProjectNature#deconfigure()
    */
   public void deconfigure() throws CoreException {
   }
   /**
    * Returns local reference to associated project 
    * 
    * @see org.eclipse.core.resources.IProjectNature#getProject()
    */
   public IProject getProject() {
      return project;
   }
   /**
    * Saves local reference to associated project.
    * 
    * @see org.eclipse.core.resources.IProjectNature#setProject(IProject)
    */
   public void setProject(IProject value) {
      project = value;
   }

}