package com.ibm.lab.soln.contributions;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import java.util.StringTokenizer;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IEditorActionDelegate;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.editors.text.TextEditor;

/**
 * A Workbench toolbar action is defined that counts the words in the 
 * selected area of a text file. See Chapter 18 for a discussion of 
 * the implementation.
 * 
 * @see org.eclipse.ui.IEditorActionDelegate
 */
public class TextEditorWordCountAction implements IEditorActionDelegate {
	static final String WORD_DELIMITERS = " .,'\"/?<>;:[]{}\\|`~!@#$%^&*()-_+=\n\r";
	TextEditor textEditor;

	/**
   * Saves a reference to the current active editor
	 * @see org.eclipse.ui.IEditorActionDelegate#setActiveEditor(IAction, IEditorPart)
	 */
	public void setActiveEditor(IAction action, IEditorPart targetEditor) {
		textEditor = (TextEditor) targetEditor;
	}

	/**
   * Counts the words in the selected text and displays the result in a 
   * dialog
	 * @see org.eclipse.ui.IActionDelegate#run(IAction)
	 */
	public void run(IAction action) {
		IDocument document =
			textEditor.getDocumentProvider().getDocument(
				textEditor.getEditorInput());
		ITextSelection ts =
			(ITextSelection) textEditor.getSelectionProvider().getSelection();
		int tokenCount;

		try {
			String text = document.get(ts.getOffset(), ts.getLength());
			tokenCount =
				new StringTokenizer(text, WORD_DELIMITERS).countTokens();
		} catch (BadLocationException e) {
			tokenCount = 0;
		}

		MessageDialog.openInformation(
			null,
			"Word Count",
			"Number of words: " + tokenCount);
	}

	/**
	 * Enables the action if text has been selected, otherwise, the action
   * is disabled.
   * @see org.eclipse.ui.IActionDelegate#selectionChanged(IAction, ISelection)
	 */
	public void selectionChanged(IAction action, ISelection selection) {
		if (selection != null && selection instanceof ITextSelection) {
			ITextSelection ts = (ITextSelection) selection;

			if (ts.getLength() == 0) {
				System.out.println("TextEditorWordCountAction disabled");
				action.setEnabled(false);
			} else {
				System.out.println("TextEditorWordCountAction enabled");
				action.setEnabled(true);
			}
		} else {
			System.out.println("TextEditorWordCountAction disabled");
			action.setEnabled(false);
		}
	}
}