package com.ibm.lab.soln.contributions;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

/**
 * This action is the implementation of an object contribution on a file 
 * resource that is enabled only if the selected file is not in-sync 
 * with the local file system; a little smarter version of the standard
 * Refresh action. The special child tag "adaptable" in the extension 
 * point org.eclipse.ui.popupMenus is handled in the selectionChanged method.
 * This action is dependent on the host plug-in to include the extension
 * point org.eclipse.ui.startup.
 * 
 * See Chapter 18 for a discussion of the implementation.
 * @see org.eclipse.ui.IObjectActionDelegate
 */

public class SmartRefreshAction implements IObjectActionDelegate {
	IWorkbenchPart targetPart;
	IFile selectedFiles[] = null;

	/**
	 * @see org.eclipse.ui.IObjectActionDelegate#setActivePart(IAction, IWorkbenchPart)
	 */
	public void setActivePart(IAction action, IWorkbenchPart tgtPart) {
		targetPart = tgtPart;
	}

	/**
	 * @see org.eclipse.ui.IActionDelegate#run(IAction)
	 */
	public void run(IAction action) {

		for (int i = 0; i < selectedFiles.length; i++) {
			IFile file = selectedFiles[i];

			if (!file.isSynchronized(IResource.DEPTH_ZERO)) {
				try {
					file.refreshLocal(
						IResource.DEPTH_ZERO,
						new NullProgressMonitor());
				} catch (CoreException e) {
				}
			}
		}
	}

	/**
   * Captures the view's selected IFile items including those items 
   * that are adaptable to an IFile object. This allows the action to 
   * operate on items in the Java Package Explorer view which contains 
   * Java classes are not files but are adaptable to files.
   * 
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(IAction, ISelection)
	 */
	public void selectionChanged(IAction action, ISelection selection) {
		ArrayList newFiles = new ArrayList();
		IFile file = null;

		if (selection != null && selection instanceof IStructuredSelection) {
			IStructuredSelection ss = (IStructuredSelection) selection;

			for (Iterator iter = ss.iterator(); iter.hasNext();) {
				Object obj = iter.next();

				if (obj instanceof IFile) {
					file = (IFile) obj;
				} else if (obj instanceof IAdaptable) {
					IAdaptable a = (IAdaptable) obj;
					IResource res = (IResource) a.getAdapter(IResource.class);

					if (res instanceof IFile)
						file = (IFile) res;
				}

				if (file != null
					&& !file.isSynchronized(IResource.DEPTH_ZERO)) {
					newFiles.add(file);
				}
			}
		}

		if (newFiles.isEmpty()) {
			selectedFiles = null;
			action.setEnabled(false);
		} else {
			selectedFiles = (IFile[]) newFiles.toArray(new IFile[newFiles.size()]);
			action.setEnabled(true);
		}

	}
}