package com.ibm.lab.soln.contributions;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowPulldownDelegate;

/**
 * A contribution to a toolbar pull-down menu is defined whose 
 * contents are based on the open workbench editors. 
 * See Chapter 18 for a discussion of the implementation.
 * @see org.eclipse.ui.IWorkbenchWindowActionDelegate
 */
public class ActivateEditorPulldownDelegate
	implements IWorkbenchWindowPulldownDelegate {

	IWorkbenchWindow currentWorkbenchWindow;

	public ActivateEditorPulldownDelegate() {
	}

	/**
	 * This method does all the work by dynamically creating a Menu and 
   * populating it with a set of MenuItem objects, each of which holds a 
   * reference to an open editor. A selection listener is created for each 
   * menu item. When a MenuItem object is selected, its corresponding 
   * editor is brought to the top and made the active editor. Observe that
   * that this method traverses all the open workbench pages, where a page 
   * maps to a Perspective. On each page, any references to open 
   * editors as IEditorReference objects are added to the menu. 
   * 
   * Note that in Eclipse 2.0, there is no longer more than one instance 
   * of an editor on the same object so it is not necessary to traverse 
   * all the workbench pages. The first loop could be eliminated
   * and replaced with the method IWorkbenchWindow.getActivePage to find 
   * all the editor references.
   * @see org.eclipse.ui.IWorkbenchWindowPulldownDelegate#getMenu(Control)
	 */
	public Menu getMenu(Control parent) {
		Menu menu = new Menu(parent);

		IWorkbenchPage pages[] = currentWorkbenchWindow.getPages();
		for (int i = 0; i < pages.length; i++) {
			IEditorReference editorRefs[] = pages[i].getEditorReferences();
			
			for (int j = 0; j < editorRefs.length; j++) {
				MenuItem menuitem = new MenuItem(menu, SWT.NONE);
				menuitem.setText(editorRefs[j].getTitle());
				menuitem.setData(editorRefs[j]);

				menuitem.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						IEditorReference editorRef =
							(IEditorReference) e.widget.getData();
						IWorkbenchPart editorPart = editorRef.getPart(true);
						
						if (editorPart != null) {
							currentWorkbenchWindow.setActivePage(
								editorPart.getSite().getPage());
							currentWorkbenchWindow.getActivePage().bringToTop(
								editorPart);
						}
					}

				});
			}
		}

		return menu;
	}

	/**
	 * Unlike most actions, the run method for this class does nothing. It 
   * is replaced by the code in the selection listener of the getMenu method.
   * @see org.eclipse.ui.IWorkbenchWindowActionDelegate#run
   * @see ActivateEditorPulldownDelegate#getMenu
	 */
	public void run(IAction action) {
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchWindowActionDelegate#selectionChanged
	 */
	public void selectionChanged(IAction action, ISelection selection) {
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchWindowActionDelegate#dispose
	 */
	public void dispose() {
	}

	/**
	 * @see org.eclipse.ui.IWorkbenchWindowActionDelegate#init
	 */
	public void init(IWorkbenchWindow window) {
		currentWorkbenchWindow = window;
	}
}
