/*
 * "The Java Developer's Guide to Eclipse"
 * 
 * (C) Copyright International Business Machines Corporation, 2003.
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

package com.ibm.lab.soln.jface;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

/**
 * @author PatMc
 *
 */
public class JFaceComponent {

	private TableViewer viewer = null;
	/**
	 * 
	 */
	public JFaceComponent() {
		super();
	}

	/**
	 * @param composite
	 */
	// Edu-Sol: 05b - Parent composite used to create viewer 
	public JFaceComponent(Composite composite) {

		//		this.composite = composite;
		this.viewer = new TableViewer(composite);
	}
	/**
	 * @param composite, table
	 */
	//	Edu-Sol: 05c - Table used to create viewer - table has a parent composite
	public JFaceComponent(Table table) {
		this.viewer = new TableViewer(table);
	}

	/**
	 * Viewer gets a content provider and label provider
	 */
	//	Edu-Sol: 06b - Viewer configured with a content and label provider
	public void configureViewer() {
		setContentProvider(new ViewContentProvider());
		setLabelProvider(new ViewLabelProvider());
	}

	/**
	 * The content provider class is responsible for providing objects to the view. It can wrap
	 * existing objects in adapters or simply return objects as-is. These objects may be sensitive
	 * to the current input of the view, or ignore it and always show the same content 
	 * (like Task List, for example).
	 */

	class ViewContentProvider
		implements IStructuredContentProvider, IResourceChangeListener {

		private IResource input;

		/**
		 * Constructor for ViewContentProvider.
		 */
		public ViewContentProvider() {
			super();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		//	Edu-Sol: 07b - Content Provider informed of input
		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
			System.out.println(
				"Viewer-> ContentProvider.inputChanged() - Input: " + newInput);

			// If this is the first time we have been given an input
			if (oldInput == null) {
				IResource resource = (IResource) newInput;
				//	Edu-Sol: 07c - Content Provider hook to model changes
				ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
			}
			this.input = (IResource) newInput;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
			System.out.println("In content provider dispose");
			ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		//		 Edu-Sol: 08  - Content Provider accesses model for data
		public Object[] getElements(Object parent) {
			System.out.println(
				"Viewer-> ContentProvider.getElements() - for parent: " + parent);

			return getViewElements();
		}

		/**
		 * Returns the members of the selected container, if not a container, then a message is returned.
		 * @return
		 */
		private Object[] getViewElements() {
			if (input instanceof IContainer) {
				IContainer inputContainer = (IContainer) input;
				IResource[] members = null;
				try {
					members = inputContainer.members();
				} catch (CoreException e) {
					// Auto-generated catch block
					e.printStackTrace();
				}
				if (members.length != 0) {
					return (Object[]) members;
				};
			}
			return new String[] { "no members to display" };
		}

		/* (non-Javadoc)
		 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
		 */
		//	 Edu-Sol: 10a - Content Provider reacts to model changes
		public void resourceChanged(IResourceChangeEvent event) {
			//	System.out.println("resourceChanged");

			final IResourceDelta delta = event.getDelta();

			// Edu-Sol: 10b - Content Provider tells viewer about model change
			Control ctrl = viewer.getControl();
			if (ctrl != null && !ctrl.isDisposed()) {
				// Do a sync exec, not an async exec, since the resource delta
				// must be traversed in this method.  It is destroyed
				// when this method returns.
				ctrl.getDisplay().syncExec(new Runnable() {
					public void run() {
						processResourceChangeDelta(delta);
					}
				});
			}
		}

		/**
		 * @param delta
		 */
		protected void processResourceChangeDelta(IResourceDelta delta) {
			// Lazy update - needs to be better
			// Should jump to IContainer that in the input 
			// and only react to changes if they exist 

			// consider viewer.update or viewer.refresh(IResource-object-element)

			viewer.refresh();

		}
	}

	/**
	 *
	 */
	class ViewLabelProvider extends LabelProvider
		implements ILabelProvider, ITableLabelProvider {
			
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object obj, int index) {
			// If an IResource then process for name
			if (obj instanceof IResource) {
				IResource res = (IResource) obj;
				if (index == 0)
					return res.getName();
				else
					return res.toString();
			}
			return getText(obj);
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object obj, int index) {
			if (index == 0)
				return getImage(obj);
			else
				return null;
		}
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
		 */
		public Image getImage(Object obj) {
			if (obj instanceof IResource) {
				IResource res = (IResource) obj;
				switch (res.getType()) {
					case IResource.PROJECT :
						return PlatformUI.getWorkbench().getSharedImages().getImage(
							ISharedImages.IMG_OBJ_PROJECT);
					case IResource.FOLDER :
						return PlatformUI.getWorkbench().getSharedImages().getImage(
							ISharedImages.IMG_OBJ_FOLDER);
					case IResource.FILE :
						return PlatformUI.getWorkbench().getSharedImages().getImage(
							ISharedImages.IMG_OBJ_FILE);
				}
			}
			return null;
		}
	}

	/**
	 * @param object
	 */
	public void setInput(Object object) {
		//      input = object;
		viewer.setInput(object);
	}

	/**
	 * @param provider
	 */
	public void setContentProvider(IContentProvider provider) {
		viewer.setContentProvider(provider);
	}

	/**
	 * @param labelProvider
	 */
	public void setLabelProvider(IBaseLabelProvider labelProvider) {
		viewer.setLabelProvider(labelProvider);
	}

	/**
	 * @param listener
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener) {
		viewer.addSelectionChangedListener(listener);
	}

	/**
	 * 
	 */
	public void refresh() {
		viewer.refresh();
	}

	/**
	 * @param element
	 * @param properties
	 */
	public void update(Object element, String[] properties) {
		viewer.update(element, properties);
	}

	/**
	 * @param elements
	 * @param properties
	 */
	public void update(Object[] elements, String[] properties) {
		viewer.update(elements, properties);
	}
	/**
	 * @return
	 */
	public TableViewer getViewer() {
		return viewer;
	}

}
