package com.ibm.lab.soln.extensionpoint;

/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import java.util.ArrayList;
import java.util.List;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPluginRegistry;
import org.eclipse.core.runtime.Platform;

/**
 * Processes the subtags of the 
 * <code>com.ibm.lab.soln.extensionpoint.toolAction</code> extension point.
 * 
 * <p>See the plugin.xml for example usage.
 */

public class ExtensionProcessor {
	// Plug-in ID from <plugin> tag in plugin.xml
	private static final String PLUGIN_ID =
		"com.ibm.lab.soln.extensionpoint";
	// Extension point ID from the <extension-point> tag in plugin.xml
	private static final String EXTENSION_POINT_ID = "toolAction";
	// Name of the tag to process
	private static final String EXTENSION_TAG = "tool";
	// Name of the attribute holding the class to do an action
	private static final String EXTENSION_TAG_ACTION_ATTRIB = "action";
	// Name of the attribute holding the label for the action
	private static final String EXTENSION_TAG_LABEL_ATTRIB = "label";

	private IPluginRegistry pluginRegistry = Platform.getPluginRegistry();
	private IExtensionPoint point =
		pluginRegistry.getExtensionPoint(PLUGIN_ID, EXTENSION_POINT_ID);
	private List toolActionsParameters = new ArrayList();

	public ExtensionProcessor() {
		super();
	}

	/**
	 * Returns the <code>ToolActionParameters</code> representing
	 * the extensions to the 
	 * <code>com.ibm.lab.soln.extensionpoint.toolAction</code> 
	 * extension point.
	 * 
	 * @return list of <code>ToolActionParameters</code>.
	 * @see 	com.ibm.lab.soln.extensionpoint.ToolActionParameters
	 */

	protected List getToolActionsParameters() {
		return toolActionsParameters;
	}

	private void addMenuItem(IConfigurationElement configElement) {
		System.out.print("Loading tag...");

		try {
			// create an instance of the class specified in the action attribute
			IToolAction toolAction =
				(IToolAction) configElement.createExecutableExtension(
					EXTENSION_TAG_ACTION_ATTRIB);

			// get menu item label
			String label = configElement.getAttribute(EXTENSION_TAG_LABEL_ATTRIB);

			// search for double entries
			boolean doubleEntry = false;
			for (int i = 0; i < getToolActionsParameters().size(); i++) {
				ToolActionParameters toolActionParameters =
					(ToolActionParameters) getToolActionsParameters().get(i);
				if (toolActionParameters.getActionLabel().equals(label))
					doubleEntry = true;
			}

			// we take the first matching label
			if (!doubleEntry) {
				getToolActionsParameters().add(new ToolActionParameters(toolAction, label));
				System.out.println("...success!");
			} else {
				System.out.println(
					"...failed! Reason: Label '"
						+ label
						+ "' already exists.  Check your plugin.xml");
			}

			System.out.println(
				"Details: "
					+ EXTENSION_TAG_ACTION_ATTRIB
					+ ": "
					+ configElement.getAttribute(EXTENSION_TAG_ACTION_ATTRIB)
					+ ", "
					+ EXTENSION_TAG_LABEL_ATTRIB
					+ ": "
					+ configElement.getAttribute(EXTENSION_TAG_LABEL_ATTRIB));
		} catch (CoreException e) {
			// this exception occurs mostly when the class attribute value
			// could not be instantiated (maybe a typing error in class name)
			System.out.print("...failed! Reason: " + e.getLocalizedMessage());
		}

	}

	/**
	 * Process extensions to the <code>com.ibm.lab.soln.extensionpoint</code>
	 * extension point.
	 * 
	 * @see	org.eclipse.core.runtime.IExtension
	 * @see	org.eclipse.core.runtime.IConfigurationElement
	 */
	public void load() {
		System.out.println("*Action Configuration Start*");

		// check: Any <extension> tags for our extension-point?
		if (point != null) {
			IExtension[] extensions = point.getExtensions();

			for (int i = 0; i < extensions.length; i++) {
				IExtension currentExtension = extensions[i];
				IConfigurationElement[] tags = currentExtension.getConfigurationElements();

				for (int j = 0; j < tags.length; j++) {
					IConfigurationElement currentTag = tags[j];

					// just load the tag we want, all others are ignored
					if (currentTag.getName().equals(EXTENSION_TAG))
						addMenuItem(currentTag);
				}
			}
		}

		// Check if no extensions or empty extensions
		if (point == null || getToolActionsParameters().size() == 0) {
			System.out.println("* No configuration found!");
		}

		System.out.println("*Action Configuration End*");
	}
}