package com.ibm.lab.soln.jdt.extras;

/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IMember;
import org.eclipse.jdt.core.ISourceRange;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.ToolFactory;
import org.eclipse.jdt.core.compiler.IScanner;
import org.eclipse.jdt.core.compiler.ITerminalSymbols;
import org.eclipse.jdt.core.compiler.InvalidInputException;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPart;

/**
 * Abstract superclass for actions wishing to delete/add <code>IMember</code> modifiers.
 */
public abstract class ChangeIMemberFlagAction implements IObjectActionDelegate {

	IMember[] members = new IMember[0];
	protected boolean isChecked = false;

	/* non-Javadoc
	 * @see org.eclipse.ui.IObjectActionDelegate#setActivePart(IAction, IWorkbenchPart)
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
	}

	/**
	 * @see org.eclipse.ui.IActionDelegate#run(IAction)
	 */
	public void run(IAction action) {
		for (int i = 0; i < members.length; i++) {
			if (canPerformAction(action, members[i]))
				performAction(action, members[i]);
		}
	}

	/**
	 * Perform the action against the given <code>IMember</code>; subclasses must override.
	 */
	abstract protected boolean performAction(IAction action, IMember member);

	/**
	 * Return <code>true</code> if the given <code>IMember</code>
	 * conforms to the action modifier; subclasses must override.
	 */
	abstract protected boolean isChecked(IAction action, IMember member);
	
	/**
	 * Return <code>true</code> if the action can be applied to the
	 * given <code>IMember</code>; can be overridden, but must call superclass.
	 */
	protected boolean canPerformAction(IAction action, IMember member) {
		return !member.isBinary() && !member.isReadOnly();
	}

	/**
	 * Replace modifier indicated by <code>replacingFlag</code> 
	 * with <code>newFlag</code>.
	 * 
	 * @param member#member to be modified
	 * @param newFlag#Flags.AccPublic et al
	 * @param replacingFlag#Flags.AccPrivate et al
	 * @param replacingTokenName#ITerminalSymbols.TokenNameprivate et al 
	 */
	protected boolean replaceFlag(
		IMember member,
		int newFlag,
		int replacingFlag,
		int replacingTokenName) {

		try {
			if ((member.getFlags() & newFlag) != 0)
				return true;

			ICompilationUnit cu = member.getCompilationUnit();

			if (cu.isWorkingCopy()) {
				IBuffer buffer = cu.getBuffer();
				IScanner scanner =
					ToolFactory.createScanner(false, false, false, false);
				scanner.setSource(buffer.getCharacters());
				ISourceRange sr = member.getSourceRange();
				scanner.resetTo(
					sr.getOffset(),
					sr.getOffset() + sr.getLength() - 1);

				if ((member.getFlags() & replacingFlag) != 0) {
					int token = scanner.getNextToken();
					while (token != ITerminalSymbols.TokenNameEOF
						&& token != ITerminalSymbols.TokenNameLPAREN
						&& token != replacingTokenName)
						token = scanner.getNextToken();

					if (token == replacingTokenName) {
						buffer.replace(
							scanner.getCurrentTokenStartPosition(),
							scanner.getCurrentTokenEndPosition()
								- scanner.getCurrentTokenStartPosition()
								+ 1,
							Flags.toString(newFlag));
						cu.reconcile();
						return true;
					}
				}
			}
		} catch (JavaModelException e) {
		} catch (InvalidInputException e) {
		}

		return false;
	}

	/**
	 * Delete modifier indicated by <code>flag</code> and  
	 * <code>flagTokenName</code>.
	 * 
	 * @param member#member to be modified
	 * @param flag#Flags.AccPublic et al
	 * @param flagTokenName#ITerminalSymbols.TokenNameprivate et al 
	 */
	protected boolean deleteFlag(IMember member, int flag, int flagTokenName) {

		try {
			if ((member.getFlags() & flag) == 0)
				return true;

			ICompilationUnit cu = member.getCompilationUnit();

			if (cu.isWorkingCopy()) {
				IBuffer buffer = cu.getBuffer();
				IScanner scanner =
					ToolFactory.createScanner(false, false, false, false);
				scanner.setSource(buffer.getCharacters());
				ISourceRange sr = member.getSourceRange();
				scanner.resetTo(
					sr.getOffset(),
					sr.getOffset() + sr.getLength() - 1);

				int token = scanner.getNextToken();
				while (token != ITerminalSymbols.TokenNameEOF
					&& token != ITerminalSymbols.TokenNameLPAREN
					&& token != flagTokenName)
					token = scanner.getNextToken();

				if (token == flagTokenName) {
					buffer.replace(
						scanner.getCurrentTokenStartPosition(),
						scanner.getCurrentTokenEndPosition()
							- scanner.getCurrentTokenStartPosition()
							+ 2,
						"");
					cu.reconcile();
					return true;
				}
			}

		} catch (JavaModelException e) {
		} catch (InvalidInputException e) {
		}

		return false;
	}
	
	/**
	 * Insert modifier indicated by <code>newFlag</code>.
	 * 
	 * @param member#member to be modified
	 * @param newFlag#Flags.AccPublic et al
	 */
	protected boolean insertFlag(IMember member, int newFlag) {

		try {
			ICompilationUnit cu = member.getCompilationUnit();

			if (cu.isWorkingCopy()) {
				IBuffer buffer = cu.getBuffer();
				IScanner scanner =
					ToolFactory.createScanner(false, false, false, false);
				scanner.setSource(buffer.getCharacters());
				ISourceRange sr = member.getSourceRange();
				scanner.resetTo(
					sr.getOffset(),
					sr.getOffset() + sr.getLength() - 1);

				scanner.getNextToken();
				buffer.replace(
					scanner.getCurrentTokenStartPosition(),
					0,
					Flags.toString(newFlag) + " ");
				cu.reconcile();

				return true;
			}
		} catch (JavaModelException e) {
		} catch (InvalidInputException e) {
		}

		return false;
	}

	/**
	 * Determine if the current action applies to the <code>IMember</code>s, given
	 * their current state. If the selection is homogeneous, also check the
	 * action if all conform to this action's modifier.
	 * 
	 * @see org.eclipse.jdt.core.IMember
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(IAction, ISelection)
	 */
	public void selectionChanged(IAction action, ISelection selection) {
		boolean canPerformAction = false;

		if (selection instanceof IStructuredSelection
			&& !selection.isEmpty()) {
			IStructuredSelection ss = (IStructuredSelection) selection;
			members = (IMember[]) ss.toList().toArray(new IMember[ss.size()]);

			isChecked = true;

			for (int i = 0; i < members.length && isChecked; i++)
				isChecked = isChecked(action, members[i]);

			canPerformAction = true;
			for (int i = 0; i < members.length && canPerformAction; i++)
				canPerformAction = canPerformAction(action, members[i]);
		} else {
			members = new IMember[0];
		}

		action.setChecked(isChecked);
		action.setEnabled(canPerformAction);
	}
}
