package com.ibm.lab.soln.jdt;

/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */
 
import java.util.*;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.*;
import org.eclipse.jdt.core.dom.*;
import org.eclipse.jdt.ui.*;
import org.eclipse.jface.action.*;
import org.eclipse.jface.viewers.*;
import org.eclipse.ui.*;

/**
 * The purpose of <code>AddTraceStatementsEditorAction</code> is to demonstrate how to 
 * add simple code modifications ("refactoring") to the Java editor. This action
 * adds <code>System.out.println</code> statements for each method, including
 * all parameters, to the beginning of each method.
 * 
 * @see	org.eclipse.jdt.core.IBuffer
 * @see	org.eclipse.jdt.core.IWorkingCopy
 * @see	org.eclipse.jdt.core.JavaCore
 * @see	org.eclipse.jdt.ui.IWorkingCopyManager
 * @see	org.eclipse.jdt.core.ICompilationUnit
 */
public class AddTraceStatementsEditorActionDelegate
	implements IEditorActionDelegate {

	IEditorPart cuEditor;

	private class JavaMethodCollector extends ASTVisitor {
		public List methodDeclarations = new ArrayList();

		public JavaMethodCollector(ICompilationUnit cu) {
			AST.parseCompilationUnit(cu, false).accept(this);
		}

		public boolean visit(MethodDeclaration node) {
			methodDeclarations.add(0, node);
			return false;
		}

		public List getMethodDeclarations() {
			return methodDeclarations;
		}
	}

	public AddTraceStatementsEditorActionDelegate() {
	}

	/**
	 * Traverse all the <code>ICompilationUnit</code>'s methods using an AST,
	 * calculating the position of each, and adding <code>System.out.println</code>
	 * for each method parameter.
	 * 
	 * @see IEditorActionDelegate#run
	 */
	public void run(IAction action) {
		IWorkingCopyManager manager = JavaUI.getWorkingCopyManager();
		IEditorInput editorInput = cuEditor.getEditorInput();
		
		try {
			manager.connect(editorInput);
		} catch (CoreException e) {
			e.printStackTrace(System.err);
			return;			
		}
		
		ICompilationUnit cu = manager.getWorkingCopy(cuEditor.getEditorInput());
		List methodDeclarations = new JavaMethodCollector(cu).getMethodDeclarations();

		try {
			IBuffer buffer = cu.getBuffer();

			// Note the methods are processed in reverse order (bottom to top)
			// in order to avoid invalidating the previously-calculated
			// AST source code start positions as new text is inserted 
			// (i.e., inserting text at the bottom of the buffer doesn't 
			// affect positions higher up).

			for (Iterator iterator = methodDeclarations.iterator(); iterator.hasNext();) {
				MethodDeclaration methodDeclaration = (MethodDeclaration) iterator.next();
				Block block = methodDeclaration.getBody();

				if (block != null) {
					int i = block.getStartPosition();
					if (i >= 0) {
						StringBuffer sb = new StringBuffer();
						sb.append("\n\t\tSystem.out.println(\"Start ");
						sb.append(methodDeclaration.getName().getIdentifier());
						sb.append("\"");

						if (!methodDeclaration.parameters().isEmpty()) {
							for (Iterator parmIterator = methodDeclaration.parameters().iterator();
									parmIterator.hasNext(); ) {
								sb.append("\n\t\t\t + \" <\" + ");
								SingleVariableDeclaration parm =
									(SingleVariableDeclaration) parmIterator.next();
								sb.append(parm.getName().getIdentifier() + " + \">\"");
							}
						}

						sb.append(");");
						buffer.replace(i+1, 0, sb.toString());
					}
				}
			}
			
			synchronized (cu) {
				cu.reconcile();
			}
		} catch (JavaModelException e) {
			e.printStackTrace(System.err);
		}
		finally {
			manager.disconnect(editorInput);
		}
	}

	/* non-Javadoc
	 * @see IEditorActionDelegate#selectionChanged
	 */
	public void selectionChanged(IAction action, ISelection selection) {
	}

	/* non-Javadoc
	 * @see IEditorActionDelegate#setActiveEditor
	 */
	public void setActiveEditor(IAction action, IEditorPart targetEditor) {
		cuEditor = targetEditor;
	}
}