package com.ibm.lab.msseditor.core;

/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.util.ListenerList;

/**
 * Simple model for a "mini-spreadsheet" containing text/integer cell values.
 * 
 * @see MiniSpreadsheetRow
 * @see IMiniSpreadsheetListener
 */

public class MiniSpreadsheet {
	public static final int MAX_COLUMN_COUNT = 10;	
	public static final int DEFAULT_COLUMN_COUNT = 4;
	public static final int DEFAULT_ROW_COUNT = 10;	
	
	private ListenerList listeners = new ListenerList();
	private MiniSpreadsheetRow[] rows;
	private int columnCount = DEFAULT_COLUMN_COUNT;
	
	/**
	 * Return a mini-spreadsheet having dimensions [rows,columns].
	 */
	public MiniSpreadsheet(int rowCount, int columnCount) {
		List list = new ArrayList(rowCount);
		this.columnCount = Math.min(columnCount, MAX_COLUMN_COUNT);
		for (int i = 0; i < rowCount; i++) {
			list.add(new MiniSpreadsheetRow(this, this.columnCount, i));
		}
		list.toArray(rows = new MiniSpreadsheetRow[list.size()]);
	}

	/**
	 * Set the rows based on the content of the provided semi-colon delimited
	 * input stream.
	 */
	public MiniSpreadsheet(InputStream is) throws CoreException {
		load(is);
	}

	/**
	 * Set the rows based on the content of the provided semi-colon delimited
	 * input stream.
	 */	
	public void load(InputStream is) throws CoreException {
		setRows(loadRows(is));
		fireRowsChanged();
	}

	/**
	 * Get the value at [row,column] as a string.
	 */
	public String getString(int row, int column) {
		return rows[row].getString(column);
	}

	/**
	 * Get the value at [row,column] as an integer or zero if not
	 * a value integer format.
	 */
	public int getInt(int row, int column) {
		return rows[row].getInt(column);
	}	

	/**
	 * Set the value at [row,column] to a string.
	 */
	public void setData(int row, int column, String newValue) {
		rows[row].setData(column, newValue);
		fireCellChanged(row, column, newValue);
	}

	/**
	 * Return the rows.
	 */
	public MiniSpreadsheetRow[] getRows() {
		return rows;
	}
	
	/**
	 * Add the listener.
	 * 
	 * @see IMiniSpreadsheetListener
	 */
	public void addMiniSpreadsheetListener(IMiniSpreadsheetListener listener) {
		listeners.add(listener);
	}

	/**
	 * Remove the listener.
	 * 
	 * @see IMiniSpreadsheetListener
	 */
	public void removeMiniSpreadsheetListener(IMiniSpreadsheetListener listener) {
		listeners.remove(listener);
	}
	
	/** 
	 * Save the current contents in a semi-colon delimited text file.
	 */	
	public void save(IFile file) throws CoreException  {
		MiniSpreadsheetRow[] msrs = getRows();
		StringBuffer sb = new StringBuffer();

		for (int i = 0; i < msrs.length; i++) {
			for (int j = 0; j < msrs[i].getColumnCount(); j++) {
				sb.append(msrs[i].getString(j) + ";");
			}
			sb.append("\n");
		}
		sb.append("\n");			
		file.setContents(new ByteArrayInputStream(sb.toString().getBytes()), 
				IResource.KEEP_HISTORY, null);
	}
	
	/**
	 * Insert a new row at the end.
	 */
	public void appendRow() {
		MiniSpreadsheetRow[] msrs = new MiniSpreadsheetRow[rows.length + 1];

		for (int i = 0; i < rows.length; i++) {
			msrs[i] = rows[i];
		}
		msrs[rows.length] = new MiniSpreadsheetRow(this, getColumnCount(), rows.length);
		setRows(msrs);
	}
	
	/**
	 * Return the number of rows.
	 */
	public int getRowCount() {
		return getRows().length;
	}

	/**
	 * Returns the number of columns.
	 */
	public int getColumnCount() {
		return columnCount;
	}

	/**
	 * Set all rows/columns to an empty string.
	 */
	public void clearAll() {
		for (int i=0; i < getRowCount(); i++) {
			for (int j=0; j < getColumnCount(); j++) {
				setData(i, j, new String());
			}
		}
	}
	
	/**
	 * Return the total of all integer cells.  Invalid cells are treated as zero.
	 */
	public int getTotal() {
		int total = 0;
		for (int i=0; i < getRowCount(); i++) {
			for (int j=0; j < getColumnCount(); j++) {
				total += getInt(i, j);
			}
		}		
		return total;
	}
	
		protected void fireCellChanged(int row, int column, String value) {
		Object[] l = listeners.getListeners();
		for (int i = 0; i < l.length; ++i) {
			((IMiniSpreadsheetListener) l[i]).valueChanged(this, row, column, value);
		}
	}

	private void setRows(MiniSpreadsheetRow[] msrs) {
		rows = msrs;
		fireRowsChanged();
	}
	
	private void fireRowsChanged() {
		Object[] l = listeners.getListeners();
		for (int i = 0; i < l.length; ++i) {
			((IMiniSpreadsheetListener) l[i]).rowsChanged(this);
		}
	}

	private MiniSpreadsheetRow[] loadRows(InputStream is) throws CoreException {
		List rows = new ArrayList();
		try {
			LineNumberReader lnr = new LineNumberReader(new InputStreamReader(is));
			
			int rowIndex = 0;

			String line = lnr.readLine();
			while (line != null && line.lastIndexOf(";") > 0) {
				StringBuffer sb = new StringBuffer(line);
				List columns = new ArrayList();
				int start = 0;
				int end = 0;
				while (end < line.length()) {
					if (line.charAt(end) == ';') {
						String token = line.substring(start, end);
						columns.add(token);
						start = end + 1;
					}
					end++;
				}
				MiniSpreadsheetRow msr =
					new MiniSpreadsheetRow(this, columns.size(), rowIndex);
				for (int i = 0; i < columns.size(); i++) {
					msr.setData(i, (String) columns.get(i));
				}
				rows.add(msr);
				rowIndex++;
				line = lnr.readLine();
			}
			is.close();

			MiniSpreadsheetRow[] msrs;
			rows.toArray(msrs = new MiniSpreadsheetRow[rows.size()]);
			if (rows.size() == 0)
				columnCount = DEFAULT_COLUMN_COUNT;
			else
				columnCount = msrs[0].getColumnCount();			

			return msrs;

		} catch (FileNotFoundException e) {
		} catch (IOException e) {
			throw new CoreException(new 
				Status(Status.ERROR, 
				"com.ibm.lab.soln.minispreadsheet", 1, e.getLocalizedMessage(), e));
		}

		return null;
	}
	
	protected void removeRow(MiniSpreadsheetRow msr) {
		MiniSpreadsheetRow[] msrs = new MiniSpreadsheetRow[rows.length - 1];
		int rowIndex = 0;
		int j = 0;
		for (int i = 0; i < rows.length; i++) {
			if (msr == rows[i]) {
				rowIndex = i;
			} else {
				msrs[j] = rows[i];
				msrs[j].setRowIndex(j);
				j++;
			}
		}

		setRows(msrs);
	}
}
