package com.ibm.lab.soln.dialogs;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */
 
import org.eclipse.core.runtime.Preferences;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

/**
 * Custom preference page implementation.  Uses the <code>Plugin</code> API to
 * access the <code>Preferences</code> object for storing preference values.
 * <p>
 * Preference keys are defined in the <code>IDialogsIDs</code> interface.
 * </p><p>
 * Defaults for preference key values are defined as part of the Dialogs plug-in.
 * </p>
 * 
 * @see DialogsPlugin#initializeDefaultPreferences
 * @see IDialogsIDs
 * @see org.eclipse.jface.preference.PreferencePage
 * @see org.eclipse.core.runtime.Preferences
 */
public class MyPrefPageControl
  extends PreferencePage
  implements IWorkbenchPreferencePage {

  // switch to control write of trace data
  private boolean traceEnabled = false;

  // Handle to preferences obtained from plugin
  private Preferences prefs;

  // widgets used in preference page to define preference values
  //   private Button myToggle;
  private Text textInfo;
  private Button listenerButton;
  private Button removeButton;
  private Button saveButton;

  /**
   * Default constructor.
   */
  public MyPrefPageControl() {
  }

  /**
   * Initializes the preference page by getting a local reference to the 
   * <code>Preferences</code> object from the <code>DialogsPlugin</code>.
   * 
   * @see IWorkbenchPreferencePage#init(IWorkbench)
   */
  public void init(IWorkbench workbench) {
    // Get a handle to the DialogsPlugin
    DialogsPlugin myPlugin = DialogsPlugin.getDefault();

    this.noDefaultAndApplyButton();

  }

  /**
   * Implement the user interface for the preference page. Returns a  
   * control that should be used as the main control for the page.
   * <p>
   * User interface defined here adds enable/disable support for a 
   * change listener that reports preference setting changes and the
   * ability to force a save of the current preference store.
   * </p>
   * @see PreferencePage#createContents(Composite)
   */

  protected Control createContents(Composite parent) {

    // Add layer to parent widget
    Composite composite = new Composite(parent, SWT.NONE);
    // Define laout rules for widget placement
    GridLayout gridLayout = new GridLayout();
    gridLayout.numColumns = 1;
    gridLayout.horizontalSpacing = 10;
    gridLayout.verticalSpacing = 10;
    composite.setLayout(gridLayout);

    GridData data = new GridData();
    data.verticalAlignment = GridData.FILL;
    data.horizontalAlignment = GridData.FILL;
    data.grabExcessHorizontalSpace = true;
    composite.setLayoutData(data);

    // Add group and contained widgets for Recent Edits prference setting
    Group group = new Group(composite, SWT.NONE);
    group.setText("Soln: Dialog Preferences Interaction");
    GridData groupData = new GridData(GridData.GRAB_HORIZONTAL);
    group.setLayoutData(groupData);

    GridLayout groupLayout = new GridLayout();
    groupLayout.numColumns = 1;
    groupLayout.horizontalSpacing = 5;
    groupLayout.verticalSpacing = 35;
    groupLayout.marginHeight = 10;
    groupLayout.marginWidth = 10;
    group.setLayout(groupLayout);

    addPrefListenerToggle(group);

    addSavePrefsButton(group);

    // return the widget used as the base for the user interface
    return composite;
  }

  /**
   * Adds a toggle button to the user interface and implements logic
   * to add and remove a change listener that will report preference
   * setting changes.
   * 
   * @param group
   */
  public void addPrefListenerToggle(Group group) {
    listenerButton = new Button(group, SWT.CHECK);
    listenerButton.setText(
      "Report modifications using a Preferences Change Listener");
    
    listenerButton.setSelection(
      DialogsPlugin.PreferenceListener.hasPrefListener());
    
    // Event logic to trigger add or remove of preference change listener
    listenerButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent event) {
        if (((Button) event.widget).getSelection())
          DialogsPlugin.PreferenceListener.addListener();
        else
          DialogsPlugin.PreferenceListener.removeListener();
      }
    });
  }

  /**
   * Adds a button to the user interface and implements logic
   * to save the plug-in's preference store when selected. 
   *  
   * @param group
   */
  public void addSavePrefsButton(Group group) {
    saveButton = new Button(group, SWT.PUSH);
    saveButton.setText("Save com.ibm.soln.dialogs Preference Store");
    // Event logic to react to button selection
    saveButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent event) {
    
        DialogsPlugin.getDefault().savePluginPreferences();
        resultInformation(
          "Save Preference Store Request",
          "Preference store for the com.ibm.soln.dialogs plug-in saved.");
      }
    });
  }
  /**
   * Uses a <code>MessageDialog</code> to show action results.
   * 
   * @see org.eclipse.jface.dialogs.MessageDialog
   */
  public void resultInformation(String title, String msg) {
    // Confirm Result
    Shell shell =
      DialogsPlugin
        .getDefault()
        .getWorkbench()
        .getActiveWorkbenchWindow()
        .getShell();
    MessageDialog.openInformation(shell, title, msg);
  }

  /**
   * Write trace statements.  
   * System.out.println with used for simplicity.
   */
  private void traceMsg(String msg) {
    if (IDialogsIDs.TRACE_ENABLED | traceEnabled)
      System.out.println(msg);
  }

}