package com.ibm.lab.soln.dialogs;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.runtime.Preferences;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

/**
 * Custom preference page implementation.  Uses the <code>Plugin</code> API
 * to access the <code>Preferences</code> object for 
 * storing preference values.
 * <p>
 * Preference keys are defined in the <code>IDialogsIDs</code> interface.
 * </p><p>
 * Defaults for preference key values are defined as part of the 
 * Dialogs plug-in.
 * </p>
 * 
 * @see DialogsPlugin#initializeDefaultPreferences
 * @see IDialogsIDs
 * @see org.eclipse.jface.preference.PreferencePage
 * @see org.eclipse.core.runtime.Preferences
 */
public class MyPrefPageBasic
  extends PreferencePage
  implements IWorkbenchPreferencePage {

  // switch to control write of trace data
  private boolean traceEnabled = false;

  // Handle to preferences obtained from plugin
  private Preferences prefs;

  // widgets used in preference page to define preference values
  //   private Button myToggle;
  private Text textInfo;
  private List limitSelection;
  private Button transientSetting;

  /**
   * Default constructor.
   */
  public MyPrefPageBasic() {
  }

  /**
   * Initializes the preference page by getting a local reference to the 
   * <code>Preferences</code> object from the <code>DialogsPlugin</code>.
   * 
   * @see IWorkbenchPreferencePage#init(IWorkbench)
   */
  public void init(IWorkbench workbench) {
    // Get a handle to the DialogsPlugin
    DialogsPlugin myPlugin = DialogsPlugin.getDefault();
    // Set local reference to Preferences object obtained from DialogsPlugin
    prefs = myPlugin.getPluginPreferences();
    traceMsg("PropCycle: " + "> in init");

    // Alternative API would set the preference store for the page using the 
    // PreferenceStore reference obtained from the DialogsPlugin.
    //      setPreferenceStore(myPlugin.getPreferenceStore());

  }

  /**
   * Implement the user interface for the preference page. Returns a  
   * control that should be used as the main control for the page.
   * <p>
   * User interface defined here supports the definition of preference settings
   * used by the Recent Edits marker creation and management logic.
   * </p>
   * @see PreferencePage#createContents(Composite)
   */

  protected Control createContents(Composite parent) {

    traceMsg("PropCycle: " + "> in createContents");
    // Add layer to parent widget
    Composite composite = new Composite(parent, SWT.NONE);
    // Define laout rules for widget placement
    GridLayout compositeLayout = new GridLayout();
    compositeLayout.numColumns = 1;
    compositeLayout.horizontalSpacing = 10;
    compositeLayout.verticalSpacing = 10;
    compositeLayout.marginHeight = 10;
    compositeLayout.marginWidth = 10;
    composite.setLayout(compositeLayout);

    GridData compositeData = new GridData(GridData.FILL_BOTH);
    compositeData.grabExcessHorizontalSpace = true;
    composite.setLayoutData(compositeData);

    // Add group and contained widgets for Recent Edits preference setting
    Group group = new Group(composite, SWT.NONE);
    group.setText("Recent Edits Setting");

    GridData groupData = new GridData(GridData.GRAB_HORIZONTAL);
    group.setLayoutData(groupData);

    GridLayout groupLayout = new GridLayout();
    groupLayout.numColumns = 1;

    groupLayout.horizontalSpacing = 30;
    groupLayout.verticalSpacing = 10;
    groupLayout.marginHeight = 15;
    groupLayout.marginWidth = 15;
    group.setLayout(groupLayout);

    Composite comp2 = new Composite(group, SWT.NONE);

    GridData comp2Data = new GridData(GridData.GRAB_HORIZONTAL);
    comp2.setLayoutData(comp2Data);

    GridLayout comp2Layout = new GridLayout();
    comp2Layout.numColumns = 2;

    comp2Layout.horizontalSpacing = 15;
    comp2Layout.verticalSpacing = 10;
    comp2Layout.marginHeight = 15;
    comp2Layout.marginWidth = 15;
    comp2.setLayout(comp2Layout);

    limitSelection = new List(comp2, SWT.BORDER);
    limitSelection.setItems(IDialogsIDs.REDIT_OPTIONS);
    int curValue = prefs.getInt(IDialogsIDs.REDIT_PREFERENCES_KEY);
    String[] curValueString = new String[] { String.valueOf(curValue)};
    limitSelection.setSelection(curValueString);

    Label labelN = new Label(comp2, SWT.NONE);
    labelN.setText(
      ": Select the number of files to keep in the    \n  Recent Edits marker list");

    transientSetting = new Button(comp2, SWT.CHECK | SWT.RIGHT);
    transientSetting.setSelection(prefs.getBoolean(IDialogsIDs.REDIT_PREFERENCES_KEY2));

    Label labelTS = new Label(comp2, SWT.NONE);
    labelTS.setText(": Select to save markers in the workspace at shut down");

    Label label = new Label(group, SWT.NONE);
    label.setText("  Changes are registered the next time the workbench is started.");

    Label labelT = new Label(composite, SWT.NONE);
    labelT.setText("Enter a text value you want saved \nin a preference key:");

    // Add text input field to support saving a string preference value
    textInfo = new Text(composite, SWT.SINGLE | SWT.BORDER);
    textInfo.setText(prefs.getString(IDialogsIDs.TEXT_PREFERENCES_KEY));
    GridData textData = new GridData(GridData.FILL_HORIZONTAL);
    textInfo.setLayoutData(textData);

    // return the widget used as the base for the user interface
    return composite;
  }
  /**
   * Method triggered as page is entered and exited. Overridden here to add
   * <code>traceMsg</code> support to help visualize the processing for a 
   * preference page.
   * 
   * @see org.eclipse.jface.dialogs.DialogPage#setVisible
   */
  public void setVisible(boolean visible) {
    super.setVisible(visible);
    traceMsg("PropCycle: " + "> in setVisible(" + visible + ")");
  }

  /**
   * Used to retrieve and set defaults for the preferences that
   * are displayed as part of this page. Defaults for preference key values 
   * are defined as part of the Dialogs plug-in.
   * 
   * @see DialogsPlugin#initializeDefaultPreferences
   * @see org.eclipse.jface.preference.PreferencePage#performDefaults
   */
  protected void performDefaults() {
    super.performDefaults();

    traceMsg("PropCycle: " + "> in performDefaults");

    // get active plug-in (default or defined)
    DialogsPlugin myPlugin = DialogsPlugin.getDefault();
    traceMsg("MBPrefPage: Current Plug-in: \n\t" + myPlugin.getClass());

    // Set user interface to the default value for the associated preference key
    textInfo.setText(prefs.getDefaultString(IDialogsIDs.TEXT_PREFERENCES_KEY));

    int defValue = prefs.getDefaultInt(IDialogsIDs.REDIT_PREFERENCES_KEY);
    String[] defValueString = new String[] { String.valueOf(defValue)};
    limitSelection.setSelection(defValueString);

  }

  /**
   * Saves preference values for the current page. This method is triggered when 
   * either the Apply or Ok push button has been selected.
   * 
   * @see org.eclipse.jface.preference.PreferencePage#performOk
   */
  public boolean performOk() {

    traceMsg("PropCycle: " + "> in performOk");

    prefs.setValue(IDialogsIDs.TEXT_PREFERENCES_KEY, textInfo.getText());

    int curSel = limitSelection.getSelectionIndex();
    if (curSel != -1)
      prefs.setValue(
        IDialogsIDs.REDIT_PREFERENCES_KEY,
        IDialogsIDs.REDIT_OPTIONS[curSel]);

    prefs.setValue(IDialogsIDs.REDIT_PREFERENCES_KEY2, transientSetting.getSelection());

    return super.performOk();
  }

  /**
   * Can be used to implement any special processing, such as notification,
   * if required. Logic to actually change preference values should be in the
   * <code>performOk</code> method as that method will also be triggered 
   * when the Apply push button is selected.
   * <p>
   * If others are interested in tracking preference changes they can use the
   * <code>addPropertyChangeListener</code> method available for for an 
   * <code>IPreferenceStore</code> or <code>Preferences</code>.
   * </p>
   * @see org.eclipse.jface.preference.PreferencePage#performApply
   * @see org.eclipse.jface.preference.IPreferenceStore#addPropertyChangeListener
   * @see org.eclipse.core.runtime.Preferences#addPropertyChangeListener
   * @see org.eclipse.jface.util.IPropertyChangeListener
   */
  protected void performApply() {
    super.performApply();
    traceMsg("PropCycle: " + "> in performApply");
  }

  /**
   * Can be used to implement any cleanup processing that may be required.
   * 
   * @see org.eclipse.jface.preference.PreferencePage#performCancel
   */
  public boolean performCancel() {
    traceMsg("PropCycle: " + "> in performCancel");
    return super.performCancel();
  }
  /**
   * Write trace statements.  
   * System.out.println with used for simplicity.
   */
  private void traceMsg(String msg) {
    if (IDialogsIDs.TRACE_ENABLED | traceEnabled)
      System.out.println(msg);
  }

}