package com.ibm.lab.newWizard.rcl;
/*
 * "The Java Developer's Guide to Eclipse"
 *   by Shavor, D'Anjou, Fairbrother, Kehn, Kellerman, McCarthy
 * 
 * (C) Copyright International Business Machines Corporation, 2003. 
 * All Rights Reserved.
 * 
 * Code or samples provided herein are provided without warranty of any kind.
 */

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.runtime.CoreException;

/**
 * Resource change listener that will create trace statements for all events
 * received.  The <code>IResourceChangeEvent</code> is processed with a 
 * visitor that will find and trace the <code>IResourceDelta</code> included 
 * in the event.
 * 
 * This class supports basic event tracing.
 */
public class ResourceChangeReporter implements IResourceChangeListener {
	// switch to control write of trace data
	private boolean traceEnabled = true;
	// Value used to prefix trace messages
	String rclMode = null;
	String rclModeDefault = "RCE: ";
	// Value used to indicate how many times the RCL has been 
	// invoked. Used as part of the prefix in trace messages.
	int rclCount = 0;
	/**
	 * Default constructor which sets a default mode value.
	 * Mode is used to tag messages to the console to help visualize the 
	 * trace data generated.
	 */
	public ResourceChangeReporter() {
		rclMode = rclModeDefault;
	}
	/**
	 * Alternate constructor to set an alternate prefix mode that will be 
	 * used for console message tagging.
	 */
	public ResourceChangeReporter(String mode) {
		this();
		rclMode = mode;
	}

	/**
	 * Notifies this listener that some resource changes 
	 * are happening, or have already happened.
	 * <p>
	 * The supplied event gives details. This event object (and the
	 * resource delta within it) is valid only for the duration of
	 * the invocation of this method.
	 * </p>
	 * <p>
	 * Note: This method is called by the platform; it is not intended
	 * to be called directly by clients.
	 * <p>
	 * Note that during resource change event notification, further changes
	 * to resources may be disallowed.
	 * </p>
	 * 
	 * @see org.eclipse.core.resources.IResourceDelta
	 *
	 * @param event the resource change event
	 */
	public void resourceChanged(IResourceChangeEvent event) {
		rclCount++;
		traceMsg("\n" + rclMode + rclCount + " -> Event triggered...");

		switch (event.getType()) {
			case IResourceChangeEvent.PRE_CLOSE :
				traceMsg(
					" -> Project "
						+ event.getResource().getFullPath()
						+ "  is about to close.");
				break;

			case IResourceChangeEvent.PRE_DELETE :
				traceMsg(
					" -> Project "
						+ event.getResource().getFullPath()
						+ "  is about to be deleted.");
				break;

			case IResourceChangeEvent.PRE_AUTO_BUILD :
				//     traceMsg(" -> Auto build about to run.");
				//     traceMsg(" -> Lets visit the delta...");
				//        try {
				//           event.getDelta().accept(
				//           new ResourceDeltaPrinter(rclMode + rclCount));
				//        } catch (CoreException e) {
				//           // See if the exception has anything to say...
				//           e.printStackTrace();
				//        }
				break;

			case IResourceChangeEvent.POST_AUTO_BUILD :

				// Only trace for this event type if not in resource change listener mode
				// This change reporter is also used by WorkspaceSaveParticipant
				if (rclMode != rclModeDefault) {
					traceMsg(" -> Auto build complete.");
					traceMsg(" -> Lets visit the delta...");
					try {
						event.getDelta().accept(new ResourceDeltaPrinter(rclMode + rclCount));
					} catch (CoreException e) {
						// See if the exception has anything to say...
						e.printStackTrace();
					}
				}
				break;

			case IResourceChangeEvent.POST_CHANGE :
				traceMsg(" -> Resource has been changed.");
				//        traceMsg(" -> Lets visit the delta...");
				try {
					IResource res = event.getResource();

					event.getDelta().accept(new ResourceDeltaPrinter(rclMode + rclCount));
				} catch (CoreException e) {
					// See if the exception has anything to say...
					e.printStackTrace();
				}
				break;
		}
	}

	/**
	 * Write trace statements.  
	 * System.out.println with prefix tagging used for simplicity.
	 */
	void traceMsg(String msg) {
		if (traceEnabled)
			System.out.println(rclMode + rclCount + msg);
	}

}