/* Traversal.java
 * Bryan Chadwick :: 2007
 * Normal parametrized traversal "function" */

package edu.neu.ccs.demeterf.perform;

import edu.neu.ccs.demeterf.dispatch.DBEntry;
import java.lang.reflect.*;

import edu.neu.ccs.demeterf.FC;
import edu.neu.ccs.demeterf.Control;
import edu.neu.ccs.demeterf.Fields;


/** Traverses an Object structure using a Builder and an Augmentor. */
public class Traversal extends AbstTraversal{
    /* Builder and Augmentor */
    protected FC func;
    protected MethodDB<Method> buildDB;
    protected MethodDB<Method> augDB;

    /** Create a onestep Traversal with the given functionality */
    public static Traversal onestep(FC f){ return new Traversal(f,Control.nowhere()); }

    /** Create a parameterized Traversal that goes Everywhere */
    public Traversal(FC f){ this(f, Control.everywhere()); }
    /** Create a Traversal with Selective edge/field Bypassing */
    public Traversal(FC f, Control c){
        super(c);
        func = f;
        buildDB = MethodDB.createMethodDB(f.getClass(), FC.buildMethodName);
        augDB = MethodDB.createMethodDB(f.getClass(), FC.augMethodName);
    }
    
    /** Apply the Builder to this list of 'Fields' */    
    protected Object applyBuilder(Object o[], boolean prim)
    { return applyFObj(func, o, buildDB, FC.buildMethodName, prim?0:-1); }

    /** Apply the Augmentor to the Argument at this Object before
     *    traversing the given field. */
    static Class[] emptyClass = new Class[0];
    protected Object applyAugment(Object o[], Class<?> pc, String fn){
        Object field = Fields.any;
        if(pc != null){
            // Look for a class that represents the field
            String fName = pc.getName()+"$"+fn;
            try{
                Class<?> fc = Class.forName(fName);
                fc.getConstructor(emptyClass).setAccessible(true);
                field = fc.newInstance();
            }catch(Exception e){ throw new RuntimeException(e); }
        }
        return applyFObj(func, new Object[]{o[0],field,o[1]},augDB, FC.augMethodName, 2);
    }

    /** Generic function object application (Faster) */
    public static Object applyFObj(Object f, Object o[], MethodDB<Method> db, String meth, int def){
        int len = o.length;
        Class<?> cs[] = new Class[len];
        for(int i = 0; i < len; i++){
            if(o[i] == null){
                throw new RuntimeException(" NULL REF: "+o[0].getClass()+" field # "+i);
            }
            cs[i] = o[i].getClass();
        }

        DBEntry<Method> ml = db.matchEntryFast(cs,len);
        if(ml == null)
            if(def < 0)
                throw new RuntimeException("\n  DemeterF: Did Not Find a Match for: \n      "+
                        Help.signature(f.getClass(), meth, cs, o.length)+"\n");
            else return o[def];

        Method m = ml.getMethod();
        Object ret = def >= 0?o[def]:null;
        try{
            if(!m.isAccessible())m.setAccessible(true);
            ret = m.invoke(f, objectSubset(o, ml.numArgs()));
        }catch(Exception e){ throw new RuntimeException(e); }
        return ret;
    }

    /** Object Array from another Object Array */
    public static Object[] objectSubset(Object o[], int len){
        if(o.length == len)return o;
        Object os[] = new Object[len];
        for(int i = 0; i < len; i++)
            os[i] = o[i];
        return os;
    }
    static class Help{
        /** Signature Printing Helper */
        public static String signature(Class<?> c, String name, Class<?> cs[], int max){
            String[] ss = new String[cs.length];
            for(int i = 0; i < max; i++)ss[i] = cs[i].getSimpleName();
            return signature(c, name, ss, max);
        }
        /** Signature Printing Helper */
        public static String signature(Class<?> c, String name, String cs[], int max){
            String s = c.getSimpleName()+"."+name+"(";
            for(int i = 0; i < max; i++){
                s += (cs == null)?"null":cs[i];
                if(i < max-1)s += (", ");
            }
            return s+")";
        }
    }
}
