package edu.neu.ccs.demeterf.lib;

import java.util.Comparator;

import edu.neu.ccs.demeterf.Fields;

/** Class representing Map Entries (key/value Pairs).
 *    Use <tt>Entry.create(Key, Val)</tt> and <tt>Entry.create(Key, Val, Comparator<Key>)</tt>
 *    to create Entries when needed. You shouldn't really need to create them, unles you want
 *    to build a Map from an array or a List. */
public class Entry<Key,Val> implements Comparable<Entry<Key,Val>>{
    protected final Key key;
    protected final Val val;
    private final Comparator<Key> comp;
    
    /** Create an Entry from a Key/Value Pair*/
    public Entry(Key k, Val v){ this(k,v,new CComp<Key>()); }
    private Entry(Key k, Val v, Comparator<Key> c){
        key = k; val = v; comp = c; 
    }
    
    /** Compare this Entry to the given one... using the inner Comparator */
    public int compareTo(Entry<Key, Val> e) { return comp.compare(key,e.key); }
    
    /** Simple Comparator for comparable elements */
    static class CComp<Key> implements Comparator<Key>{
        public int compare(Key a, Key b){ return ((Comparable<Key>)a).compareTo(b); }
    }
    
    /** Simple Comparator for Sorting Entries */
    static class LComp<K,V> extends List.Comp<Entry<K,V>>{
        Comparator<K> c;
        LComp(Comparator<K> cc){ c = cc; }
        public boolean comp(Entry<K,V> a, Entry<K,V> b){
            return c.compare(a.getKey(), b.getKey()) <= 0;
        }
    }
    /** Simple Comparator for Entry/Key Equality */
    static class EComp<K,V> extends List.Comp<Entry<K,V>>{
        Comparator<K> c;
        EComp(Comparator<K> cc){ c = cc; }
        public boolean comp(Entry<K,V> a, Entry<K,V> b){
            return c.compare(a.getKey(), b.getKey()) == 0;
        }
    }
    /** Simple Comparator for Entry/Key Equality */
    static class KPred<K,V> extends List.Pred<Entry<K,V>>{
        K key;
        Comparator<K> c;
        KPred(K k, Comparator<K> cc){ key = k; c = cc; }
        public boolean huh(Entry<K,V> e){
            return c.compare(key, e.getKey()) == 0;
        }
    }
    
    /** Field Class for key */
    public static class key extends Fields.any{}
    /** Field Class for val */
    public static class val extends Fields.any{}
    
    /** Getter for Entry.key */
    public Key getKey(){ return key; }
    /** Getter for Entry.val */
    public Val getVal(){ return val; }
    
    /** Return a String representation of this Entry.  Essentially "(Key -> Val)"*/
    public String toString(){ return "("+key+" -> "+val+")"; }
    /** Create an Entry with a Comparable Key, and a Value. */
    public static <Key extends Comparable<Key>,Val> Entry<Key,Val> create(Key k, Val v)
    { return new Entry<Key,Val>(k,v,new CComp<Key>()); }
    /** Create an Entry with a Key, Value, and a Comparator for Keys. */
    public static <Key,Val> Entry<Key,Val> create(Key k, Val v, Comparator<Key> c)
    { return new Entry<Key,Val>(k,v,c);  }

    /** Standard Equality */
    public boolean equals(Object o){
        if(!(o instanceof Entry))return false;
        return ((comp.compare(key, ((Entry<Key,Val>)o).key) == 0) &&
                val.equals(((Entry<Key,Val>)o).val));
    }
    
    /** Return the HashCode of this Entry */
    public int hashCode(){ return key.hashCode()+3*val.hashCode(); }
}
