package edu.neu.ccs.demeterf.lexer;

import edu.neu.ccs.demeterf.demfgen.Diff;
import edu.neu.ccs.demeterf.demfgen.Preamble;
import edu.neu.ccs.demeterf.lib.*;
import edu.neu.ccs.demeterf.lexer.classes.*;
import edu.neu.ccs.demeterf.util.CLI;
import edu.neu.ccs.demeterf.*;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.PrintStream;

/** Main Class for DemFLex Generator */
public class DemFLex{
    static String
    graph = "--graph",
    nocompress = "--nocompress",
    quiet = "--quiet",
    nowarn = "--nowarn";

    static void p(String s){ System.err.print(s); }
    /** Main Lexer Generation method */
    public static void main(String[] args) throws Exception{
        List<String>[] all = CLI.splitArgs(args);
        List<String> opts = all[CLI.OPTS];
        List<String> nonopts = all[CLI.ARGS];
        String nonOpt[] = nonopts.toArray(new String[nonopts.length()]);
        Diff.storeOptions(opts);
    
        if(Diff.optionSet(Diff.help))usage(true);
        if(nonOpt.length != 2)usage(false);
        
        String
           inFile = nonOpt[0],
           outFile = nonOpt[1];
        
        try{
            LexMain mn = LexMain.parse(new FileInputStream(inFile));
            genLex(mn,opts,outFile);
        }catch(Exception re){
            p(" Error:\n   "+re.getMessage().replace("\n","\n    ")+"\n");
        }            
    }
    public static void genLex(LexMain mn, List<String> opts, String outFile) throws Exception{
        long start = System.currentTimeMillis();
        boolean loud = !Diff.optionSet(quiet);
        
        PrintStream out = new PrintStream(new FileOutputStream(outFile));
            if(loud)p(" ** Rewriting\n");
            List<TokDef> ts = rewriteAll(mn.getDs(),Map.<ident,RegExp>create());
            if(loud)p(" ** Building NFAs\n");
            List<NFA> nfas = convert(ts);
            final NFA fnfa = nfas.fold(new List.Fold<NFA,NFA>(){
                public NFA fold(NFA f, NFA r){ return r.rawOr(f); }
            }, NFA.empty());

            List<FinalState> fins = nfas.zip(new List.Zip<NFA,TokDef,FinalState>(){
                public FinalState zip(NFA n, TokDef t){
                    State fin = n.getStart().equals(n.getFin())?fnfa.getStart():n.getFin();
                    return new FinalState(fin, ""+t.getId()); }
            }, ts);

            if(Diff.optionSet(graph)){
                out.println(nfaToGraph(fnfa,fins)+"\n");
                return;
            }

            Mach m = toDFA(fnfa);
            if(loud)p(" ** Building DFA\n");

            String cls = className(outFile);
            out.println(Preamble.header+mn.getPkg());
            out.println("public class "+className(outFile)+
                    " extends edu.neu.ccs.demeterf.lexer.Lexer{");
            out.println("    static class DFA implements edu.neu.ccs.demeterf.lexer.ADFA{");
            out.println(Diff.optionSet(nocompress)?m.transArray():m.smallTransArray());
            out.println(m.finalArray(fins, Diff.optionSet(nowarn) || !loud));
            out.println(
                    "\n\n"+
                    "        /* Access Methods */\n"+
                    "        public int EOF(){ return T_EOF; }\n"+
                    "        public int SKIP(){ return T_SKIP; }\n"+
                    "        public int[][] getEDGES(){ return EDGES; }\n"+
                    "        public int[][] getFINAL(){ return FINAL; }\n"+
                    "        public String[] getNAMES(){ return NAMES; }\n"+
                    "    }\n\n"+
                    "    /* Create a "+cls+" Lexer... */\n"+
                    "    public "+cls+"(java.io.InputStream inpt){ super(inpt, new DFA()); }\n");
            out.println("}");
            if(loud)p(" ** Finished ["+time(start)+" sec.]\n");
    }
    static String className(String file){
        int d = file.lastIndexOf(".");
        int s = file.lastIndexOf(java.io.File.separator);
        if(s < 0)s = 0; else s++;
        if(d < s)throw new LexEx("Non-Java FileName");
        return file.substring(s,d);
    }
    static void usage(boolean help){
        p("\n"+(help?"":" !! Incorrect arguments\n !!")+
                " Usage: java demflex [Options] <LEX-File> <Output-File>\n\n"+
                " The order/placement of options doesn\'t matter, but the relative\n"+
                "    order of the manditory ones must be as shown.\n\n"+
                " LEX-File contains the Lexer declarations to be generated,\n"+
                " Output-File is the Java File in which to place the generated DFA\n"+
                " Options can be: \n"+
                //12345678901234567890123456789012345678901234567890123456789012345678901234567890
                "   --help            : Print this usage information.\n"+
                "   --quiet           : Don't print Token overlap warnings.\n"+
                "   --nocompress      : Don't compress the State Table (might not compile then).\n"+
                "   --graph           : Write out a DOT/GraphViz graph of the final NFA to the\n"+
                "                         output File, instead of the DFA class.\n"+
                "\n");
        System.exit(1);
    }
    /** Simple LexGen Exception*/
    static class LexEx extends RuntimeException{
        LexEx(String s){ super(s); }
    }

    static java.text.DecimalFormat form = new java.text.DecimalFormat("0.00");
    private static String time(long s){
        double t = (System.currentTimeMillis()-s)/1000.0;
        return form.format(t);
    }
    /** Convert a List of Tokens into a List of NFAs */
    static List<NFA> convert(List<TokDef> ts){
        return new Traversal(new ID(){
            NFA combine(StrRE re){ throw new LexEx("Strings should be rewritten"); }
            NFA combine(PlusRE re){ throw new LexEx("Plus should be rewritten"); }
            NFA combine(LocRef re, ident id){ throw new LexEx("Unbound Local Def: "+id); }
            NFA combine(ChRE re, char v){ return NFA.empty().addTrans(new ChLbl(v)); }
            NFA combine(NChRE re, char v){ return NFA.empty().addTrans(new NChLbl(v)); }
            NFA combine(GrpRE re, List<GrpPart> gs)
            { return NFA.empty().addTrans(new GrpLbl(gs)); }
            NFA combine(NGrpRE re, List<GrpPart> gs)
            { return NFA.empty().addTrans(new NGrpLbl(gs)); }
            NFA combine(ConcatRE re, List<NFA> nfas){ return NFA.concat(nfas); }
            NFA combine(OrRE re, List<NFA> nfas){ return NFA.or(nfas); }
            NFA combine(StarRE re, NFA nfa){ return nfa.star(); }
            NFA combine(HuhRE re, NFA nfa){ return nfa.huh(); }
            NFA combine(TokDef t, ident id, NFA n){ return n; }
            List<NFA> combine(List<?> l, NFA n, List<NFA> r){ return r.push(n.rmdups()); }
            List<NFA> combine(Empty<?> e){ return List.<NFA>create(); }
        },Control.bypass("edu.neu.ccs.demeterf.lexer.classes.GrpRE.gs "+
                "edu.neu.ccs.demeterf.lexer.classes.NGrpRE.gs")).<List<NFA>>traverse(ts);
    }

    /** Convert the NFA to a DFA with sets of States */
    static Mach toDFA(NFA nfa){
        List<Trans> eps = nfa.epsilons();
        Set<State> start = NFA.closure(Set.create(nfa.getStart()), eps);
        Mach m = new Mach(start);
        int last = 1, curr = 1;

        while(curr <= last){
            Set<State> currState = m.state(curr);
            for(char c = 0; c < 256; c++){
                Set<State> e = nfa.DFAEdge(currState,c,eps);
                int look = m.lookup(e);
                if(look >= 0)
                    m = m.addTrans(curr, c, look);
                else{
                    last++;
                    m = m.addState(last,e).addTrans(curr, c, last);
                }
            }
            curr++;
        }
        return m;
    }

    /** Rewrite a list of complex expresstions into simpler ones... */
    static List<TokDef> rewriteAll(List<LexDef> ds, final Map<ident,RegExp> locs){
        return Traversal.onestep(new ID(){
            List<TokDef> combine(List<?> l, LocDef loc, List<LexDef> r)
            { return rewriteAll(r, locs.put(loc.getId(), rewrite(loc.getRe(),locs))); }
            List<TokDef> combine(List<?> l, TokDef tok, List<LexDef> r)
            { return rewriteAll(r, locs).push(rewrite(tok,locs)); }
            List<TokDef> combine(Empty<?> e){ return List.<TokDef>create(); }
        }).traverse(ds);
    }

    /** Function Class for flattening Nested Concat and Or RegExps */  
    static class Flat extends TP{
        List<RegExp> combine(List<?> l, List<RegExp> res, List<RegExp> r){ return r.push(res); }
    }
    /** Flatten immediate nested Concat/Or RegExps */
    static List<RegExp> flatten(List<RegExp> res, Flat f){
        return new Traversal(f,Control.bypass("edu.neu.ccs.demeterf.lib.Cons.first"))
        .<List<RegExp>>traverse(res);
    }
    /** Rewrite complex RegExps to simpler ones */
    static <T> T rewrite(T t, final Map<ident,RegExp> locs){
        return new Traversal(new TP(){
            RegExp combine(StrRE s, String val)
            { return toConcat(val, List.<RegExp>create()); }
            RegExp combine(PlusRE s, RegExp inner)
            { return new ConcatRE(List.<RegExp>create(inner, new StarRE(inner))); }
            RegExp combine(LocRef re, ident id){
                if(!locs.containsKey(id))
                    throw new LexEx("Unbound Local: "+id);
                return locs.get(id);
            }
            RegExp combine(OrRE re, List<RegExp> args){
                return new OrRE(flatten(args, new Flat(){
                    List<RegExp> combine(List<?> l, OrRE e, List<RegExp> r)
                    { return r.push(e.getRes()); }
                }));
            }
            RegExp combine(ConcatRE re, List<RegExp> args){
                return new ConcatRE(flatten(args, new Flat(){
                    List<RegExp> combine(List<?> l, ConcatRE e, List<RegExp> r)
                    { return r.push(e.getRes()); }
                }));
            }
        }).<T>traverse(t);
    }
    /** Create a RegExp from a String */
    static RegExp toConcat(String s, List<RegExp> lor){
        if(s.length() == 0){
            if(lor.length() == 1)return lor.top();
            return new ConcatRE(lor.reverse());
        }
        return toConcat(s.substring(1), lor.push(new ChRE(s.charAt(0))));
    }
    /** Create a graph (Dot/GraphViz) representation of an NFA */
    static String nfaToGraph(NFA nfa, List<FinalState> fins){
        return "digraph G{\n"+
        "  rankdir=LR;\n"+
        "  node [shape=doublecircle,style=filled,fillcolor=\"#66FF66\"];\n"+
        "    \""+nfa.getStart()+"\";\n"+
        "  node [shape=ellipse,style=filled,fillcolor=\"#FF6666\"];\n"+
        "  "+fins.toString(new List.Stringer<FinalState>(){
            public String toString(FinalState f, List<FinalState> r){
                return "  \""+f.getS()+"\"[label=\""+f.getTok().toUpperCase()+"\"]\n;";
            }
        })+
        "  node [shape=circle,style=filled,fillcolor=\"#FFFFFF\"];\n"+
        nfa.getTrans().toString(new List.Stringer<Trans>(){
            public String toString(Trans t, List<Trans> r){
                return "  \""+t.getFrm()+"\" -> \""+t.getTo()+
                "\"[label=\" "+PrintToString.escape(""+t.getL())+"\"];\n";
            }
        })+"}";
    }
}
