package edu.neu.ccs.demeterf.dispatch;
import java.lang.reflect.Method;
import java.lang.reflect.Constructor;

import edu.neu.ccs.demeterf.lib.List;

/** MethodDB is a List of DBEntries. Includes selection functions
 *    and a few creators. */
public class MethodDB<M>{
    public MethodDB(List<DBEntry<M>> m){ methods = m; }
    
    public static List<DBEntry<Method>> getMethods(Class<?> c, String name){
        List<DBEntry<Method>> l = List.<DBEntry<Method>>create();
        while(c != null && !c.equals(Object.class)){
            for(Method m:c.getDeclaredMethods())
                if(m.getName().equals(name)){
                    MethodEntry me = new MethodEntry(m);
                    if(l.index(me) < 0)
                        l = l.push(me);
                }
            c = c.getSuperclass();
        }
        return l;
    }
    public static MethodDB<Method> createMethodDB(Class<?> c, String name){
        return new MethodDB<Method>(getMethods(c,name));
    }
    public static MethodDB<Constructor<?>> createConstrDB(Class<?> c){ 
        List<DBEntry<Constructor<?>>> l = List.<DBEntry<Constructor<?>>>create();
        for(Constructor<?> con:c.getDeclaredConstructors())
            l = l.push(new ConstrEntry(con));
        return new MethodDB<Constructor<?>>(l);
    }
    static boolean debug = false;
    public static void setDebug(boolean b){ debug = b; }
    public static boolean isDebug(){ return debug; }
    protected List<DBEntry<M>> methods;
    
    /** !Quickly! return the most specific (first) Match from the entries.
     * Returns <code>null</code> if none is found. */
    public DBEntry<M> matchEntryFast(Class<?> args[]){
        return matchEntryFast(args, args.length, methods);
    }
    private static <M> DBEntry<M> matchEntryFast(Class<?> args[], int len, List<DBEntry<M>> db){
        if(db.isEmpty())return null;
        DBEntry<M> e = db.top();
        if(applicable(args,len,e))
            return matchEntryFast(args, len, db.pop(), e);
        return matchEntryFast(args, len, db.pop());
    }
    private static <M> boolean applicable(Class<?> args[], int len, DBEntry<M> e){
        if(e.numArgs() > len)return false;
        for(int i = 0; i < e.numArgs(); i++)
            if(!e.arg(i).isAssignableFrom(args[i]))return false;
        return true;
    }
    private static <M> DBEntry<M> matchEntryFast(Class<?> args[], int len,
                                                 List<DBEntry<M>> db, DBEntry<M> bst){
        if(db.isEmpty())return bst;
        DBEntry<M> e = db.top();
        if(applicable(args,len,e) && TypeSort.compare(e, bst))
            return matchEntryFast(args, len, db.pop(), e);
        return matchEntryFast(args, len, db.pop(), bst);
    }
    
    /** Match All the entries that are applicable */
    public List<DBEntry<M>> matchAllEntries(final Class<?> args[]){
        return methods.filter(new List.Pred<DBEntry<M>>(){
            public boolean huh(DBEntry<M> e){
                return applicable(args, args.length, e);
            }});
    }
    
    /** Normal toString() */
    public String toString(){ return methods.toString(); }
    /** Return a String representation of this MethodDB with <code>s</code>
     * in between the method entries */
    public String toString(String s){ return methods.toString(s,""); }
    
    /** Filter the DBEntries based on the number of arguments and the type
     *    of the first argument.  Greater than one means that they place a
     *    constraint on the traversal of an instance of "c" (if applicable) */
    public List<DBEntry<M>> prefixApp(final Class<?> c){
        return methods.filter(new List.Pred<DBEntry<M>>(){
            public boolean huh(DBEntry<M> e){
                return (e.numArgs() >= 1 &&
                        e.arg(0).isAssignableFrom(c));
            }
        });
    }
    
    public List<DBEntry<M>> allMethods(){ return methods; }
}
