/* CompTraversal.java
 * Bryan Chadwick :: 2007
 * Traversal with Functors */

package edu.neu.ccs.demeterf.compose;

import java.lang.reflect.Field;

import edu.neu.ccs.demeterf.*;
import edu.neu.ccs.demeterf.lib.List;
import edu.neu.ccs.demeterf.util.*;

/** Traverses an Object structure using <tt>Functors<tt>: combined Function Classes
 *   that know what to do. The constructors take any number of <tt>Functors</tt> or 
 *   unwrapped function objects. During traversal, the first function object is
 *   treated normally, possibly with a traversal argument. All others receive the
 *   previous Functor's result as a final combine argument.
 *   <p>See {@link edu.neu.ccs.demeterf.examples.Compose Compose} for an example.</p>
 */
public class CompTraversal{
    Control bypass;
    Functor funcs[];
    int length;
    
    /** Create a Traversal that goes Everywhere using any number of Functors */
    public CompTraversal(Functor ... fs){ this(fs, Control.everywhere()); }
    /** Create a Traversal with traversal control, using any number of Functors */
    public CompTraversal(Functor fs[], Control c){
        bypass = c; funcs = fs; length = fs.length;
    }
    
    /** Create a Traversal that goes Everywhere using any number of
     *    function objects */
    public CompTraversal(FC ... fs){ this(fs, Control.everywhere()); }
    /** Create a Traversal with traversal control using any number of
     *    function objects */
    public CompTraversal(FC fs[], Control c){
        this(new Traversal(new ID(){
            Functor combine(FC f){ return new Functor(f); }
        },Control.builtins(FC.class))
        .<Functor[]>traverse(fs), c);
    }
    
    /** Do the Traversal... No arguments, but only return the <i>last</i> result */
    public <T> T traverseLast(Object o){
        return (T)traverse(o, Option.none())[length-1];    
    }
    /** Do the Traversal... With an argument, but only return the <i>last</i> result */
    public <T> T traverseLast(Object o, Object arg){
        return (T)traverse(o, Option.some(arg))[length-1];    
    }
    /** Do the Traversal... No arguments, return <b>all</b> the results */
    public Object[] traverse(Object o){
        return traverse(o, Option.none());    
    }
    /** Do the Traversal... With an argument, return <b>all</b> the results */
    public Object[] traverse(Object o, Object a){
        return traverse(o, Option.some(a));
    }
    
    private static int indent = 0;
    private static String indent(){ return indent(indent); }
    private static String indent(int i){ if(i==0)return ""; return " "+indent(i-1); }

    /** Do the Traversal... With an ML like Option argument (Some/None)*/
    protected Object[] traverse(Object o, Option arg){
        indent++;
        Class<?> c = o.getClass();
        boolean hasArg = arg.some();
        Object result[];
        List<Field> fl = Util.getFuncFields(c);
        
        if(bypass.isBuiltIn(c)){
            result = applyF(o, arg);
        }
        else{
            Option narg = (hasArg)?Option.some(funcs[0].applyAugment(new Object[]{o, arg.get()})):arg;
            Object[][] tret = new Object[fl.length()][];
            for(int i = 0; i < fl.length(); i++){
                try{
                    Field f = fl.lookup(i);
                    Object fld = f.get(o);
                    if(fld == null){
                        if(!Util.allowNull)Util.nullFieldError(f);
                    }else{
                        if(!bypass.skip(c, f.getName()))
                            tret[i] = traverse(fld, narg);
                        else
                            tret[i] = applyF(fld, narg);
                    }
                }catch(Exception e){ throw (RuntimeException)e; }
            }
            result = applyB(o,tret,arg);
        }
        indent--;
        return result;
    }
    
    /** Apply the Functor inplace of a leaf object (BuiltIn)*/
    private Object[] applyF(Object o, Option arg){
        Object res[] = new Object[length];
        res[0] = funcs[0].applyBuilder(Util.addArg(new Object[]{o}, arg),true);
        for(int i = 1; i < length; i++)
            res[i] = funcs[i].applyBuilder(new Object[]{o,res[i-1]},true);
        return res;
    }
    /** Apply the Functor inplace of a non-leaf object (Compound)*/
    private Object[] applyB(Object o, Object [][]r, Option arg){
        Object res[] = new Object[length];
        res[0] = funcs[0].applyBuilder(make(o, r, 0, arg),false);
        for(int i = 1; i < length; i++)
            res[i] = funcs[i].applyBuilder(make(o, r, i, Option.some(res[i-1])),false);
        return res;
    }
    /** Grab all the <idx>idx</tt> results of each of the Functors, to reconstruct
     *   an <tt>Object[]</tt> to be passed to another Functor */
    private Object[] make(Object targ, Object r[][], int idx, Option arg){
        final int max = r.length+1+(arg.some()?1:0);
        Object ret[] = new Object[max];
        ret[0] = targ;
        for(int i = 0; i < r.length; i++)
            ret[i+1] = r[i][idx];
        if(arg.some())ret[max-1] = arg.get();
        return ret;
    }
 }
