; Copyright 1993 Lars Thomas Hansen.
;
; Portable Standard Scheme -- the assembler
; Instruction assembler for the Intel 80386/80486/Pentium.
;
; July 24, 1997 / lth
;
; Notable features not included (but ought to be) in order of importance:
;   - short/near jump optimization
;   - support for operand/address size prefixes
;   - 80387 instructions
;   - byte-set-on-condition
;   - string instructions
;   - scaled/based addressing modes
;
; Each instruction's bit pattern is generated by a procedure whose name
; starts with 'i386.'.  The procedure is called with arguments that
; correspond to the operands of the operation indicated by the procedure's
; name.  The first argument to the procedure is always an 'as' structure.  
; The procedure returns nothing but emits instruction bytes and fixups
; to the 'as' structure.
;
; The assembler accepts syntax in the style of MASM, where the first operand
; is the result as well as the first source operand. This is in contrast to
; e.g. GAS, which has the result and 'left' operand as the second operand to
; the instruction.
;
; Note
;   80387 instructions should be called i387.*
;   80486 instructions should be called i486.*
;   80487 instructions should be called i487.*
;   Pentium instructions should be called p5.*
;   Pentium II instructions should be called p6.*
;
; Bugs
;   There is no way to specify a 16-bit value or ptr in an expression.
;
; Operand syntax:
;
;  operand  -->  <label> | <reg> | <imm> | <mem>
;  addr     -->  <reg> | <mem>
;  label    -->  ( <n> <value> )
;  reg      -->  ( r <width> <code> )
;  imm      -->  <expr>
;  mem      -->  ( @+ reg disp )            dword ptr
;             |  ( @ reg )                  dword ptr
;             |  ( $+ reg disp )            byte ptr
;             |  ( $ reg )                  byte ptr
;  disp     -->  signed
;  signed   -->  <signed-exact-integer>
;  expr       |  (byte <expr1>)
;             |  (dword <expr1>)
;             |  <expr1>
;  expr1    -->  (+ <expr1> <expr1> ...) 
;             |  (- <expr1> <expr1> ...)
;             |  label
;             |  signed
;
; EXPERIMENTAL (970724): instead of using as.byte! a lot, we instead
; construct entire instructions.  In particular, the mod/rm-byte 
; procedure takes the following arguments:
;   as structure
;   reg/imm flag
;   operand
;   number of bytes to leave empty (unspecified) before mod/rm bytes
;   number of bytes to leave empty after mod/rm bytes
; and it returns a bytevector that will hold the entire instruction.
; This makes the code a little noisier but should make the assembler
; faster and consume less memory overall.

; Macro for fast and convenient construction of short bytevectors.

(define-syntax bytevector
  (syntax-rules ()
    ((bytevector a)
     (let ((v (make-bytevector 1)))
       (bytevector-set! v 0 a)
       v))
    ((bytevector a b)
     (let ((v (make-bytevector 2)))
       (bytevector-set! v 0 a)
       (bytevector-set! v 1 b)
       v))
    ((bytevector a b c)
     (let ((v (make-bytevector 3)))
       (bytevector-set! v 0 a)
       (bytevector-set! v 1 b)
       (bytevector-set! v 2 c)
       v))
    ((bytevector a b c d)
     (let ((v (make-bytevector 4)))
       (bytevector-set! v 0 a)
       (bytevector-set! v 1 b)
       (bytevector-set! v 2 c)
       (bytevector-set! v 3 d)
       v))
    ((bytevector a b c d e)
     (let ((v (make-bytevector 5)))
       (bytevector-set! v 0 a)
       (bytevector-set! v 1 b)
       (bytevector-set! v 2 c)
       (bytevector-set! v 3 d)
       (bytevector-set! v 4 e)
       v))))
     

; Registers.
;
; A symbolic machine register name is a list (r n k) where n is the width
; of the register in bits and k is the machine code for the register.

(define i386:eax '(r 32 #b000))
(define i386:ecx '(r 32 #b001))
(define i386:edx '(r 32 #b010))
(define i386:ebx '(r 32 #b011))
(define i386:esp '(r 32 #b100))
(define i386:ebp '(r 32 #b101))
(define i386:esi '(r 32 #b110))
(define i386:edi '(r 32 #b111))

(define i386:ax  '(r 16 #b000))

(define i386:al  '(r 8 #b000))
(define i386:ah  '(r 8 #b100))
(define i386:cl  '(r 8 #b001))
(define i386:ch  '(r 8 #b101))
(define i386:bl  '(r 8 #b011))
(define i386:bh  '(r 8 #b111))
(define i386:dl  '(r 8 #b010))
(define i386:dh  '(r 8 #b110))

(define i386-instruction)

(let ()

  ;; Registers.
  
  (define (register? op)
    (and (pair? op) (eq? (car op) 'r)))

  (define (register-width r) (cadr r))
  
  (define (register-code r) (caddr r))

  (define (reg32? r)
    (and (pair? op) (eq? (car op) 'r) (= (cadr r) 32)))

  (define (reg16? r)
    (and (pair? op) (eq? (car op) 'r) (= (cadr r) 16)))
  
  (define (reg8? r)
    (and (pair? op) (eq? (car op) 'r) (= (cadr r) 8)))

  ;; Memory addresses.
  
  (define (memop? op)
    (and (pair? op) 
         (case (car op)
           ((@ @+ $+ $) #t)
           (else #f))))

  (define (mem32? op)
    (and (pair? op) 
         (case (car op)
           ((@+ @) #t)
           (else #f))))

  (define (mem8? op)
    (and (pair? op)
         (case (car op)
           (($+ $) #t)
           (else #f))))

  ;; Immediate values.
  
  (define (simm8? n)
    (or (and (number? n) (<= -128 n 127))
        (and (pair? n) (eq? (car n) 'byte))))

  (define (uimm8? n)
    (or (and (number? n) (<= 0 n 255))
        (and (pair? n) (eq? (car n) 'byte))))

  (define (imm8? n)
    (or (simm8? n) (uimm8? n)))

  (define (uimm16? n)
    (and (<= 0 n) (<= n 65535)))        ; bogus

  (define (imm16? n)
    (and (<= -32768 n) (<= n 65535)))   ; bogus

  (define (simm32? n)
    (or (and (number? n) (<= -2147483648 n) (<= n 2147483647))
        (and (pair? n) (not (eq? (car n) 'byte)))))

  (define (uimm32? n)
    (or (and (number? n) (<= 0 n) (<= n 4294967295))
        (and (pair? n) (not (eq? (car n) 'byte)))))

  (define (imm32? n)
    (or (simm32? n) (uimm32? n)))

  (define (immediate? n)
    (and (not (regop? n)) (not (memop? n))))
  
  ;; bits

  (define reg->reg/mem 0)   ; direction bit
  (define reg/mem->reg 2)   ; direction bit
  (define size32 1)         ; word bit
  (define size8 0)          ; word bit
  (define sexyes 2)         ; sign extend bit
  (define sexno  0)         ; sign extend bit


  ;; General binary ALU operation: add, or, adc, sbb, and, sub, xor, cmp.
  ;; These all have extended operations in Group 1.
  ;;
  ;;    (op addr addr)
  ;;    (op addr imm)
  ;;
  ;; "op" is the opcode in the general (mod/rm) case; d=0, w=0
  ;; "reg/imm" is the REG value for when o2=imm and we use Group 1.
  
  (define (asm.aluop2 op reg/imm)

    (define opimm #x80)      ; Group1 basis opcode
    (define acc8  4)         ; add to opcode for AL/ib operation
    (define acc32 5)         ; add to opcode for EAX/id operation

    ;; O2 is an immediate; o1 is reg or memory.
    ;; Essentially, decide whether to use Group 1 or ACC instructions.

    (define (aluop2-imm as o1 o2)
      (cond ((and (eq? o1 i386:al) (imm8? o2))
             (let ((i (bytevector (+ op acc8) 0)))
               (dep-imm8! i 1 o2)
               (emit! as i)))
            ((and (eq? o1 i386:eax) (imm32? o2))
             (let ((i (bytevector (+ op acc32) 0 0 0 0)))
               (dep-imm32! i 1 o2)
               (emit! as i)))
            ((simm8? o2)
             (let ((i (mod/rm-byte as reg/imm o1 1 1)))
               (if (thirtytwobit? o1)
                   (bytevector-set! i 0 (+ opimm sexyes size32))
                   (bytevector-set! i 0 (+ opimm sexno size8)))
               (dep-simm8! i (- (bytevector-length i) 1) o2)
               (emit! as i)))
            ((imm32? o2)
             (let ((i (mod/rm-byte as reg/imm o1 1 4)))
               (bytevector-set! i 0 (+ opimm sexno size32))
               (dep-imm32! i (- (bytevector-length i) 4) o2)
               (emit! as i)))
            (else
             (asm-error "i386: aluop2 bogus operands" op))))

    (lambda (as o1 o2)
      (cond ((immediate? o2)
             (aluop2-imm as o1 o2))
            (else
             (let ((i (mod/rm-byte as o1 o2 1))
                   (d (if (register? o1) reg/mem->reg reg->reg/mem))
                   (s (if (eightbit? o1) size8 size32)))
               (bytevector-set! i 0 (+ op d s))
               (emit! as i))))))
  
  ;; Unary ALU operations: neg, not, inc, dec.
  ;;
  ;; "Op" is the opcode for a general r/m operand, with w=0.
  ;; "reg/imm" is the extended opcode for the REG field for the r/m operand.
  ;; "Opdwreg", if not #f, is an opcode for use with a doubleword register.
  ;;   In this case, the instruction comes out to one byte with the
  ;;   register in the low three bits of the opcode.

  (define (asm.aluop1 op reg/imm opdwreg)
    (lambda (as o1)
      (if (reg32? o1)
          (if opdwreg
              (emit! as (bytevector (+ opdwreg (regcode o1))))
              (let ((i (mod/rm-byte as reg/imm o1 1 0)))
                (bytevector-set! i 0 (+ op size32))
                (emit! as i)))
          (let ((i (mod/rm-byte as reg/imm o1 1 0)))
            (if (or (reg8? o1) (mem8? o1))
                (bytevector-set! i 0 (+ op size8))
                (bytevector-set! i 0 (+ op size32)))
            (emit! as i)))))

  ;; A byteop is just a single byte instruction with no other info.
  
  (define (asm.byteop x)
    (lambda (as)
      (emit! as (bytevector x))))

  ;; Load-effective-address is not terribly useful but is a fast 3-address
  ;; add-constant-to-reg-into-reg substitute.

  (define (asm.lea as dest addr)
    (if (and (reg32? dest) (mem32? addr))
        (let ((i (mod/rm-byte as dest addr 1 0)))
          (bytevector-set! as 0 #x80)
          (emit! as i))
        (asm-error "i386: lea: bogus:" dest addr)))

  ;; (jcc dist dest)
  ;;
  ;; Conditional branches; cc is a condition, dist is the distance 
  ;; (short, near, or best). If dist is 'best,
  ;; then the assembler is encouraged to make the best out of it.
  ;; Currently, it translates best to near.
  ;;
  ;; FIXME: here, _dist_ should be removed, and there should be heuristics
  ;; and code in the assembler to choose the best branch distance.
  
  (define (asm.jcc short near)
    (lambda (as dist dest)
      (cond ((eq? dist 'short)
             (let ((i (jmp/call-displacement8 as dest 1)))
               (bytevector-set! i 0 short)
               (emit! as i)))
            ((or (eq? dist 'near) (eq? dist 'best))
             (let ((i (jmp/call-displacement32 as dest 2)))
               (bytevector-set! i 0 #x0F)
               (bytevector-set! i 1 near)
               (emit! as i)))
            (else
             (asm-error "i386: bad jcc distance:" dist)))))

  ;; (jmp dist dest)
  ;; (jmp reg)
  ;; (jmp mem)
  ;;
  ;; Unconditional jump; dist is 'short or 'near or 'best. If dist is 'best,
  ;; then the assembler is advised to make the best out of it, otherwise
  ;; produce the requested type. If a short is requested but cannot be
  ;; done, the assembler will signal an error. Currently, 'best is the same
  ;; as 'near.
  ;;
  ;; FIXME: Again, get rid of _dist_.

  (define (asm.jump as . rest)
    (if (null? (cdr rest))
        (let ((dest (car rest)))
          (as.byte! as #xFF)
          (cond ((or (reg32? dest) (mem32? dest))
                 (emit! as (mod/rm-byte as #b100 dest 0 0)))
                (else
                 (asm-error "i386: jmp: invalid mode:" i))))
        (let ((dist (car rest))
              (dest (cadr rest)))
          (cond ((or (eq? dist 'near) (eq? dist 'best))
                 (let ((i (jmp/call-displacement32 as dest 1)))
                   (bytevector-set! as #xE9)
                   (emit! as i)))
                ((eq? dist 'short)
                 (let ((i (jmp/call-displacement8 as dest 1)))
                   (bytevector-set! as #xEB)
                   (emit! as i)))
                (else
                 (asm-error "i386: jmp: invalid dist:" dist))))))

  ;; (call label)     ; relative direct
  ;; (call mem)       ; indirect
  ;; (call reg)       ; indirect

  (define asm.call
    (lambda (as dest)
      (if (or (regop? dest) (memop? dest))
          (let ((i (mod/rm-byte as $b010 dest 1 0)))
            (bytevector-set! i 0 #xFF)
            (emit! as i))
          (let ((i (jmp/call-displacement32 as dest 1)))
            (bytevector-set! i 0 #xE8)
            (emit! as i)))))

  ;; (move r r)       ; uses r/m (88 - 8B)
  ;; (move r addr)    ; uses r/m (88 - 8B)
  ;; (move r imm)     ; optimizable opcode (B0 - BF)
  ;; (move addr r)    ; uses r/m (88 - 8B)
  ;; (move addr imm)  ; uses r/m (C6 - C7)

  (define (asm.move as op1 op2)

    (define general-rm #x88)
    (define imm-rm #xC7)
    (define fastimm8 #xB0)
    (define fastimm32 #xB8)

    (define (err)
      (asm-error "i386: bad move: mov" op1 op2))

    (cond ((mem8? op1)
           (cond ((reg8? op2)
                  (as.byte! as (+ general-rm reg->reg/mem size8))
                  (mod/rm-byte as op2 op1))
                 ((immediate? op2)
                  (as.byte! as (+ imm-rm size8))
                  (mod/rm-byte as 0 op1) ;; FIXME: reg value not doc'd
                  (immediate8 as op2))
                 (else
                  (err))))
          ((mem32? op1)
           (cond ((reg32? op2)
                  (as.byte! as (+ general-rm reg->reg/mem size32))
                  (mod/rm-byte as op2 op1))
                 ((immediate? op2)
                  (as.byte! as (+ imm-rm size32))
                  (mod/rm-byte as 0 op1)  ;; FIXME: REG value is not documented
                  (immediate32 as op2))
                 (else
                  (err))))
          ((reg8? op1)
           (cond ((or (reg8? op2) (mem8? op2))
                  (as.byte! as (+ general-rm reg/mem->reg size8))
                  (mod/rm-byte as op1 op2))
                 ((immediate? op2)
                  (as.byte! as (+ fastimm8 (regcode op1)))
                  (immediate8 as op2))
                 (else
                  (err))))
          ((reg32? op1)
           (cond ((or (mem32? op2) (reg32? op2))
                  (as.byte! as (+ general-rm reg/mem->reg size32))
                  (mod/rm-byte as op1 op2))
                 ((immediate? op2)
                  (as.byte! as (+ fastimm32 (regcode op1)))
                  (immediate32 as op2))
                 (else
                  (err))))
          (else
           (err))))

  ;; (mov*x reg32, reg8/mem8)
  ;;
  ;; Move with sign/zero extension. Supports only 8-bit srcs now.
  ;; sex? is #t if we sign-extend (movsx), #f if zero-extend (movzx).

  (define (asm.movx sex?)
    (lambda (as dest src)
      (as.byte! as #x0F)
      (as.byte! as (if sex? #xBE #xB6))
      (mod/rm-byte as dest src)))

  ;; (pop reg32)
  ;; (pop mem32)
  
  (define (asm.pop as dest)
    (cond ((reg32? dest)
           (as.byte! as (+ #x58 (regcode dest))))
          ((mem32? dest)
           (as.byte! as #x8F)
           (mod/rm-byte as 0 dest))
          (else
           (asm-error "i386: pop: not valid:" dest))))

  ;; (push reg32)
  ;; (push expr)
  ;; (push mem32)

  (define (asm.push as src)
    (display src)
    (cond ((reg32? src)
           (display "*1*")
           (as.byte! as (+ #x50 (regcode src))))
          ((mem32? src)
           (display "*2*")
           (as.byte! as #xFF)
           (mod/rm-byte as #b110 src))
          ((simm8? src)
           (as.byte! as (+ #x68 sexyes))
           (immediate8 as src))
          ((imm32? src)
           (as.byte! as (+ #x68 sexno))
           (immediate32 as src))
          (else
           (asm-error "i386: push: bad operand:" src))))

  ;; (op reg/mem, CL)   #xD2 + w /e
  ;; (op reg/mem, 1)    #xD0 + w /e
  ;; (op reg/mem, n)    #xC0 + w /e

  (define (asm.shiftop bits)
    (lambda (as src count)
      (cond ((eqv? count i386:cl)
             (if (or (reg32? src) (mem32? src))
                 (as.byte! as #xD3)
                 (as.byte! as #xD2))
             (mod/rm-byte as bits src))
            ((eqv? count 1)
             (if (or (reg32? src) (mem32? src))
                 (as.byte! as #xD1)
                 (as.byte! as #xD0))
             (mod/rm-byte as bits src))
            ((integer? count)
             (if (or (reg32? src) (mem32? src))
                 (as.byte! as #xC1)
                 (as.byte! as #xC0))
             (mod/rm-byte as bits src))
            (else
             (asm-error "i386: shiftop: bad count:" count)))))

  (define (asm.int as n)
    (cond ((= n 3)
           (as.byte! as #xCC))
          ((uimm8? n)
           (as.byte! as #xCD)
           (as.byte! as n))
          (else
           (asm-error "i386: int: bad operand" n))))

  (define (asm.ret as . rest)
    (cond ((null? rest)
           (as.byte! as #xC3))
          ((uimm16? (car rest))
           (as.byte! as #xC2)
           (immediate16 as (car rest)))
          (else
           (asm-error "i386: ret: bad operand" (car rest)))))

  ;; (xchg eax, reg32)      #x90..#x97
  ;; (xchg reg/mem, reg)    #x86 + w
  ;;
  ;; Trivia: xchg eax, eax == #x90 == nop

  (define (asm.xchg as o1 o2)
    (cond ((and (eqv? o1 i386:eax) (reg32? o2))
           (as.byte! as (+ #x90 (regcode o2))))
          ((and (eqv? o2 i386:eax) (reg32? o1))
           (as.byte! as (+ #x90 (regcode o1))))
          ((and (or (reg8? o1) (mem8? o1)) (reg8? o2))
           (as.byte! as #x86)
           (mod/rm-byte as o2 o1))
          ((and (or (reg32? o1) (mem32? o1)) (reg32? o2))
           (as.byte! as #x87)
           (mod/rm-byte as o2 o1))
          (else
           (asm-error "i386: bad xchg:" o1 o2))))

  ;; (test reg/mem, reg)   #x84 + w
  ;; (test reg/mem, imm)   #xF6 + w /0
  ;; (test acc, imm)       #xA8 + w

  (define (asm.test as o1 o2)

    (define (err)
      (asm-error "i386: test: bad ops:" o1 o2))

    (cond ((and (eqv? o1 i386:eax) (immediate? o2))
           (as.byte! as #xA9)
           (immediate32 as o2))
          ((and (eqv? o1 i386:al) (immediate? o2))
           (as.byte! as #xA8)
           (immediate8 as o2))
          ((regop? o2)
           (cond ((or (reg32? o1) (mem32? o1))
                  (as.byte! as #x85)
                  (mod/rm-byte as o2 o1))
                 ((or (reg8? o1) (mem8? o1))
                  (as.byte! as #x84)
                  (mod/rm-byte as o2 o1))
                 (else
                  (err))))
          ((immediate? o2)
           (cond ((or (reg32? o1) (mem32? o1))
                  (as.byte! as #xF7)
                  (mod/rm-byte as 0 o1)
                  (immediate32 as o2))
                 ((or (reg8? o1) (mem8? o1))
                  (as.byte! as #xF6)
                  (mod/rm-byte as 0 o1)
                  (immediate8 as o2))
                 (else
                  (err))))
          (else
           (err))))

  ;; (imul reg/mem)                    #xF6 + w /5  (hokey syntax)
  ;; (imul r32, reg/mem32)             #x0FAF
  ;; (imul r32, reg/mem32, imm8)       #x6B
  ;; (imul r32, imm8)                  #x6B (shorthand for previous)
  ;; (imul r32, reg/mem32, imm32)      #x69
  ;; (imul r32, imm32)                 #x69 (shorthand for previous)

  (define (asm.imul as . args)
    ...)

  ;; (DIV/MUL al, reg/mem8)               #xF6  /e        ax <- al op 8bit
  ;; (DIV/MUL eax, reg/mem32)             #xF7  /e   edx:eax <- eax op 32bit
  ;;
  ;; The div, idiv, and mul instructions all look the same.

  (define (asm.divmul bits)
    (lambda (as o1 o2)
      (cond ((and (eqv? o1 i386:al) (or (mem8? o2) (reg8? o2)))
             (as.byte! as #xF6)
             (mod/rm-byte as bits o2))
            ((and (eqv? o1 i386:eax) (or (mem32? o2) (reg32? o2)))
             (as.byte! as #xF7)
             (mod/rm-byte as bits o2))
            (else
              (asm-error "i386: idiv/div/mul: bad ops" o1 o2)))))


  ;; Emit a 32-bit immediate displacement relative to the address of 
  ;; the byte following the displacement. 'Dest' is an expression which
  ;; can be evaluated to a constant.

  (define (jmp/call-displacement32 as dest)
    (let ((newe (list '- dest (+ (here as) 4))))
      (signed-immediate32 as newe)))

  ;; Ditto 8-bit displacement; error if it won't fit in 8 bits.

  (define (jmp/call-displacement8 as dest)
    (let ((newe (list '- dest (+ (as.lc as) 1))))
      (signed-immediate8 as newe)))

  (define (signed-immediate8 as n)
    (immediate as 1 n simm8? as.byte!))

  (define (signed-immediate32 as n)
    (immediate as 4 n simm32? as.word!))

  (define (immediate8 as n)
    (immediate as 1 n imm8? as.byte!))

  (define (immediate16 as n)
    (immediate as 2 n imm16? as.halfword!))

  (define (immediate32 as n)
    (immediate as 4 n imm32? as.word!))

  ;; Emit an immediate (from the expression 'n') to the code stream,
  ;; or emit 0's and a fixup if this is not possible.

  (define (immediate as length n pred? out!)
    (let ((e   (eval-expr as n))
          (err (lambda () (asm-error "i386: bad immediate:" n))))
      (if (number? e)
          (if (pred? e)
              (out! as e)
              (err))
          (let ((lc (as.lc as)))
            (out! as 0)
            (as.fixup! as 
                       lc 
                       length
                       (lambda ()
                         (let ((e (eval-expr as n)))
                           (if (and (number? e) (pred? e))
                               e
                               (err)))))))))

  ;; Given a value for REG and an ADDRESS (a register or memory operand),
  ;; create a mod/rm byte with the appropriate values: REG corresponds
  ;; to the reg field, and the values for mod and r/m are chosen based
  ;; on the structure of ADDRESS.
  ;;
  ;; Should work for both 8- and 32-bit ADDRESSes.

  (define (mod/rm-byte as reg address before after)

    (define (err . msgs)
      (apply asm-error (cons "i386: mod/rm:" msgs)))

    (let ((regbits (if (register? reg)
                       (* (register-code reg) 8)
                       (* reg 8))))
      (cond ((register? address)
             (let ((i (make-bytevector (+ before after 1))))
               (bytevector-set! i before (+ #b11000000
                                            regbits 
                                            (register-code address)))
               i))
            ((memop? address)
             (let ((base   (cadr address))
                   (offset (if (null? (cddr address)) 
                               0 
                               (caddr address))))
               (cond ((eq? base i386:esp)
                      (err "You can't use ESP for a base register."))
                     ((not (number? offset))
                      (err "Non-literal-offsets not supported." offset))
                     ((and (zero? offset) (not (eq? base i386:ebp)))
                      (let ((i (make-bytevector (+ before after 1))))
                        (bytevector-set! i before (+ #b00000000 
                                                     regbits 
                                                     (regcode base)))
                        i))
                     ((simm8? offset)
                      (let ((i (make-bytevector (+ before after 2))))
                        (bytevector-set! i before (+ #b01000000 
                                                     regbits 
                                                     (regcode base)))
                        (bytevector-set! i (+ before 1) offset)
                        i))
                     ((simm32? offset)
                      (let ((i (make-bytevector (+ before after 5))))
                        (bytevector-set! i before (+ #b10000000 
                                                     regbits 
                                                     (regcode base)))
                        (dep-word! i (+ before 1) offset)
                        i))
                     (else
                      (err "Bad address:" address)))))
            (else
             (err "Bad address:" address)))))

  (set! i386-instruction
        (lambda (class . args)
          (case class
            ((aluop1) (apply asm.aluop1 args))
            ((aluop2) (apply asm.aluop2 args))
            ((byteop) (apply asm.byteop args))
            ((call)   (apply asm.call args))
            ((divmul) (apply asm.divmul args))
            ((imul)   asm.imul)
            ((int)    asm.int)
            ((jcc)    (apply asm.jcc args))
            ((jump)   asm.jump)
            ((lea)    asm.lea)
            ((move)   asm.move)
            ((movx)   (apply asm.movx args))
            ((pop)    asm.pop)
            ((push)   asm.push)
            ((ret)    asm.ret)
            ((shiftop) (apply asm.shiftop args))
            ((test)   asm.test)
            ((xchg)   asm.xchg)
            (else ???))))
  
  'i386-instruction)
            
; Instruction constructor procedures.

(define i386.adc    (i386-instruction 'aluop2 #x10 #b010))
(define i386.add    (i386-instruction 'aluop2 #x00 #b000))
(define i386.and    (i386-instruction 'aluop2 #x20 #b100))
(define i386.call   (i386-instruction 'call))
(define i386.cwde   (i386-instruction 'byteop #x98))
(define i386.cmp    (i386-instruction 'aluop2 #x38 #b111))
(define i386.cdq    (i386-instruction 'byteop #x99))
(define i386.dec    (i386-instruction 'aluop1 #xFE #b001 #x48))
(define i386.div    (i386-instruction 'divmul 6))
(define i386.idiv   (i386-instruction 'divmul 7))
(define i386.imul   (i386-instruction 'imul))
(define i386.inc    (i386-instruction 'aluop1 #xFE #b000 #x40))
(define i386.int    (i386-instruction 'int))
(define i386.into   (i386-instruction 'byteop #xCE))
(define i386.ja     (i386-instruction 'jcc #x77 #x87))
(define i386.jae    (i386-instruction 'jcc #x73 #x83))
(define i386.jb     (i386-instruction 'jcc #x72 #x82))
(define i386.jbe    (i386-instruction 'jcc #x76 #x86))
(define i386.je     (i386-instruction 'jcc #x74 #x84))
(define i386.jg     (i386-instruction 'jcc #x7F #x8F))
(define i386.jge    (i386-instruction 'jcc #x7D #x8D))
(define i386.jl     (i386-instruction 'jcc #x7C #x8C))
(define i386.jle    (i386-instruction 'jcc #x7E #x8E))
(define i386.jne    (i386-instruction 'jcc #x75 #x85))
(define i386.jno    (i386-instruction 'jcc #x71 #x81))
(define i386.jnp    (i386-instruction 'jcc #x7B #x8B))
(define i386.jns    (i386-instruction 'jcc #x79 #x89))
(define i386.jo     (i386-instruction 'jcc #x70 #x80))
(define i386.jp     (i386-instruction 'jcc #x7A #x8A))
(define i386.js     (i386-instruction 'jcc #x78 #x88))
(define i386.jmp    (i386-instruction 'jump))
(define i386.lea    (i386-instruction 'lea))
(define i386.mov    (i386-instruction 'move))
(define i386.movsx  (i386-instruction 'movx #t))
(define i386.movzx  (i386-instruction 'movx #f))
(define i386.mul    (i386-instruction 'divmul 4))
(define i386.neg    (i386-instruction 'aluop1 #xF6 #b011 #f))
(define i386.nop    (i386-instruction 'byteop #x90))
(define i386.not    (i386-instruction 'aluop1 #xF6 #b010 #f))
(define i386.or     (i386-instruction 'aluop2 #x08 #b001))
(define i386.pop    (i386-instruction 'pop))
(define i386.popa   (i386-instruction 'byteop #x61))
(define i386.popf   (i386-instruction 'byteop #x9D))
(define i386.push   (i386-instruction 'push))
(define i386.pusha  (i386-instruction 'byteop #x60))
(define i386.pushf  (i386-instruction 'byteop #x9C))
(define i386.rep    (i386-instruction 'byteop #xF3))
(define i386.repne  (i386-instruction 'byteop #xF2))
(define i386.ret    (i386-instruction 'ret))
(define i386.sal    (i386-instruction 'shiftop #b100))
(define i386.sar    (i386-instruction 'shiftop #b111))
(define i386.shr    (i386-instruction 'shiftop #b101))
(define i386.sbb    (i386-instruction 'aluop2 #x18 #b011))
(define i386.sub    (i386-instruction 'aluop2 #x28 #b101))
(define i386.test   (i386-instruction 'test))
(define i386.xchg   (i386-instruction 'xchg))
(define i386.xor    (i386-instruction 'aluop2 #x30 #b110))

; Aliases

(define i386.jc     i386:jb)
(define i386.jz     i386:je)
(define i386.jna    i386:jbe)
(define i386.jnae   i386:jb)
(define i386.jnb    i386:jae)
(define i386.jnbe   i386:ja)
(define i386.jnc    i386:jae)
(define i386.jng    i386:jle)
(define i386.jnge   i386:jl)
(define i386.jnl    i386:jge)
(define i386.jnle   i386:jg)
(define i386.jnz    i386:jne)
(define i386.jpe    i386:jp)
(define i386.jpo    i386:jnp)
(define i386.shl    i386.sal)

; eof
