
/*
 * @(#)Tester.java    28 March 28 2005
 *
 */
import edu.neu.ccs.console.*;
import java.util.*;

/**
 * 
 * @author Viera K. proulx
 *
 * test harness that catches exceptions in tests
 * uses the visitor pattern to accept test cases
 */
class Tester implements ConsoleAware{
	
	/*
	 * A complete record of all failed test results
	 */
	String testResults = "Test results: \n";
	
	/**
	 * A complete record of all test results
	 */
	String fullTestResults = "Full test results: \n";
	
	/**
	 * total number of tests
	 */
	int n;			
	
	/**
	 * total number of errors
	 */
	int errors;	
	
	/**
	 * Constructor: no tests, no errors
	 *
	 */
	Tester() {
		this.n = 0; 
		this.errors = 0;
	}
	
	/**
	 * run the tests, accept the class to be tested as a visitor
	 * 
	 * @param f the class that contains the tests
	 */
	void runTests(Testable f) {
		this.n = 0;
		try {
			f.tests(this);
		}
		catch (Throwable e) {  // catch all exceptions
			this.errors = this.errors + 1;
			console.out.println("Threw exception during test " + this.n);
			console.out.println(e);
		}
		finally {
			done();
		}
	}
	
	/**
	 * method to report the results
	 *
	 */
	public void done(){
		if (this.errors > 0)
			console.out.print("Failed " + this.errors + " out of ");
		else
			console.out.print("Passed all ");
		console.out.println (this.n + " tests.");
	}
	
	//--------- TESTS --------------------------------------------------

	/**
	 * general contractor to report test results
	 * 
	 * @param success did the test succeed?
	 * @param testname the name of this test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void report(boolean success, String testname, 
			    String expected, String actual){
		if (success)
			this.reportSuccess(testname, expected, actual);
		else
			this.reportErrors(testname, expected, actual);
	}


	/**
	 * test that only reports success or failure
	 * 
	 * @param testname the name of the test
	 * @param result the result fo the test
	 */
	void test(String testname, boolean result){
		if (!result){
			this.errors = this.errors + 1;
			this.testResults = this.testResults + 
						  	  testname + ": errors \n";
		}
		this.n = this.n + 1;
	}
	
	/**
	 * test that compares two objects using Java (or overridden) equals
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, Object expected, Object actual){
		this.report(expected.equals(actual), 
			    testname, "" + expected, "" + actual);
	}

	/**
	 * test that compares two objects using same method
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, Iterator expected, Iterator actual){
		this.report(sameIt(expected, actual), 
			    testname, "" + expected, "" + actual);
	}

	/**
	 * test that compares two objects using Java (or overridden) equals
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	// compare two structures for iterator equality
	public boolean sameIt(Iterator it1, Iterator it2){
		if (it1.hasNext() && it2.hasNext())
			return it1.next().equals(it2.next())
			    && this.sameIt(it1, it2);
		else
			return (!it1.hasNext()) && (!it2.hasNext());
	}
	
	/**
	 * test that compares two int-s using == operator
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, int expected, int actual){
		this.report(expected == actual, 
			    testname, "" + expected, "" + actual);
	}

	/**
	 * test that compares two short-s using == operator
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, short expected, short actual){		
		this.report(expected == actual, 
			    testname, "" + expected, "" + actual);
}

	/**
	 * test that compares two integers using == operator
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, long expected, long actual){	
		this.report(expected == actual, 
				    testname, "" + expected, "" + actual);
	}

	/**
	 * test that compares two boolean-s using == operator
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, boolean expected, boolean actual){		
		this.report(expected == actual, 
				    testname, "" + expected, "" + actual);
	}

	/**
	 * test that compares two char-s using == operator
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, char expected, char actual){		
		this.report(expected == actual, 
				    testname, "" + expected, "" + actual);
	}

	/**
	 * test that compares two float-s within epsilon limit
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void test(String testname, float expected, float actual, float epsilon){	
		if (Math.abs(expected - actual) < epsilon)			
			this.reportSuccess(testname, "" + expected, "" + actual);
		else
			this.reportErrors(testname, "" + expected, "" + actual);			
	}

	/**
	 * test that compares two double-s within epsilon limit
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */ 
	void test(String testname, double expected, double actual, double epsilon){
		this.report(Math.abs(expected - actual) < epsilon, 
				   testname, "" + expected, "" + actual);
	}

	//	--------- TEST REPORTS ------------------------------------------------
	
	/**
	 * record a failed test
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	void reportErrors(String testname, 
			Object expected, 
			Object actual){
		// update the count of the errors tests
		this.errors = this.errors + 1;
		
		// add test report to the errors tests report
		this.testResults = this.testResults + "\n" +
		   testname + ": errors \n" +
		   "expected: " + expected + "\n" +
		   "actual:   " + actual + "\n";
		
		// add test report to the full test report
		this.fullTestResults = this.fullTestResults + "\n" +
		   testname + ": errors \n" +
		   "expected: " + expected + "\n" +
		   "actual:   " + actual + "\n";
		
		// update the count of all tests
		this.n = this.n + 1;
	}
	
	/**
	 * record a successful test
	 * 
	 * @param testname the name of the test
	 * @param expected the expected outcome
	 * @param actual the actual outcome
	 */
	// record a successful test
	void reportSuccess(String testname, 
			Object expected, 
			Object actual){
		
		// add test report to the full test report
		this.fullTestResults = this.fullTestResults + "\n" +
		   testname + ": success \n" +
		   "expected: " + expected + "\n" +
		   "actual:   " + actual + "\n";
		
		// update the count of all tests
		this.n = this.n + 1;
	}
	
	/**
	 * report on the number and nature of errors tests
	 *
	 */
	public void testReport(){
		if (this.errors == 1){
			console.out.println(this.testResults + 
					            this.errors + " test errors.\n");
		}
		else if (this.errors > 1){
			console.out.println(this.testResults + 
		            this.errors + " tests errors.\n");
		}
		else
			console.out.println(this.testResults + 
					           "All tests passed.\n");
	}
	
	/**
	 * produce test names and values compared for all tests
	 *
	 */
	void fullTestReport(){
		if (this.errors == 1){
			console.out.println("\n 1 test errors.\n");
		}
		else if (this.errors > 1){
			console.out.println("\n " + this.errors + " tests errors.\n");
		}
		else
			console.out.println("All tests passed.\n");
		console.out.println(this.fullTestResults);
	}
	
}