/**
  Copyright 2011 Viera K. Proulx
 */

import tester.*;
import java.util.*;

/*
                      +---------------------------------+
                      | IShape                          |<----------------------------------------------------------+-+
                      +---------------------------------+                                                           | |
                      +---------------------------------+                                                           | |
                      | double area()                   |                                                           | |
                      | double distTo0()                |                                                           | |
                      | boolean biggerThan(IShape that) |                                                           | |
                      | CartPt getLoc()                 |                                                           | |
                      +---------------------------------+                                                           | |
                                     |                                                                              | |
                                    / \                                                                             | |
                                    ---                                                                             | |
                                     |                                                                              | |
                   -------------------------------------                                                            | |
                   |                                   |                                                            | |
    +---------------------------------+   +---------------------------------+   +---------------------------------+ | |
    | Circle                          |   | Square                          |   | Combo                           | | |
    +---------------------------------+   +---------------------------------+   +---------------------------------+ | |
  +-| CartPt loc                      | +-| CartPt loc                      | +-| CartPt loc                      | | |   
  | | String color                    | | | String color                    | | | IShape top                      |-+ |
  | | int radius                      | | | int size                        | | | IShape bot                      |---+
  | +---------------------------------+ | +---------------------------------+ | +---------------------------------+  
  | | double area()                   | | | double area()                   | | | double area()                   |
  | | double distTo0()                | | | double distTo0()                | | | double distTo0()                |   
  | | boolean biggerThan(IShape that) | | | boolean biggerThan(IShape that) | | | boolean biggerThan(IShape that) |
  | +---------------------------------+ | +---------------------------------+ | +---------------------------------+
  +----+   +----------------------------+                                     |
       |   |  +---------------------------------------------------------------+
       |   |  |
       v   v  v
  +--------------------------+
  | CartPt                   |
  +--------------------------+
  | int x                    |
  | int y                    |
  +--------------------------+
  | double distTo0() |
  | double distTo(CartPt pt) |
  +--------------------------+

 */

// to represent a geometric shape
interface IShape {
  // to compute the area of this shape
  public double area();

  // to compute the distance form this shape to the origin
  public double distTo0();

  // is the area of this shape is bigger than the area of the given shape?
  public boolean biggerThan(IShape that);
  
  // get the pinhole location of this shape
  public CartPt getLoc();
}

// to represent a circle
class Circle implements IShape {
  CartPt loc;
  String color;
  int radius;

  Circle(CartPt loc, String color, int radius) {
    this.loc = loc;
    this.color = color;
    this.radius = radius;
  }

  /*
    // ** TEMPLATE **
    public returnType methodName() {
    ... this.loc ...                 -- CartPt
    ... this.radius ...              -- int
    ... this.color ...               -- String

    ... this.area() ...                  -- double
    ... this.distTo0() ...               -- double
    ... this.biggerThan(AShape that) ... -- boolean
   */
  // to compute the area of this shape
  public double area(){
    return Math.PI * this.radius * this.radius;
  }

  // to compute the distance form this shape to the origin
  public double distTo0(){
    return this.loc.distTo0() - this.radius;
  }


  // is the area of this shape is bigger than the area of the given shape?
  public boolean biggerThan(IShape that){
    return this.area() >= that.area();
  }
  
  // get the pinhole location of this circle: its center
  public CartPt getLoc(){
    return this.loc;
  }
}

// to represent a square
class Square implements IShape {
  CartPt loc;
  String color;
  int size;

  Square(CartPt loc, String color, int size) {
    this.loc = loc;
    this.color = color;
    this.size = size;
  }

  /*
    // ** TEMPLATE **
    returnType methodName() {
    ... this.loc ...             -- CartPt
    ... this.size ...            -- int
    ... this.color ...           -- String

    ... this.area() ...                  -- double
    ... this.distTo0() ...               -- double
    ... this.biggerThan(AShape that) ... -- boolean
    }
   */

  // to compute the area of this shape
  public double area(){
    return this.size * this.size;
  }

  // to compute the distance form this shape to the origin
  public double distTo0(){
    return this.loc.distTo0();
  }


  // is the area of this shape is bigger than the area of the given shape?
  public boolean biggerThan(IShape that){
    return this.area() >= that.area();
  }
  
  // get the pinhole location of this square: its NW corner
  public CartPt getLoc(){
    return this.loc;
  }
}

// to represent a combination of two shapes
class Combo implements IShape{
  IShape top;
  IShape bot;
  
  Combo(IShape top, IShape bot){
    this.top = top;
    this.bot = bot;
  }

  // to compute the area of this shape
  public double area(){
    return this.top.area() + this.bot.area();
  }

  // to compute the distance form this shape to the origin
  public double distTo0(){
    return Math.min(this.top.distTo0(), this.bot.distTo0());
  }

  // is the area of this shape is bigger than the area of the given shape?
  public boolean biggerThan(IShape that){
    return this.area() >= that.area();
  }
  
  // get the pinhole location of this combo: the pinhole for the bottom
  public CartPt getLoc(){
    return this.bot.getLoc();
  }
}

/*
  +--------+
  | CartPt |
  +--------+
  | int x  |
  | int y  |
  +--------+

 */

// to represent a Cartesian point
class CartPt {
  int x;
  int y;

  CartPt(int x, int y) {
    this.x = x;
    this.y = y;
  }
  
  Random rand = new Random();

  // to compute the distance form this point to the origin
  public double distTo0(){
    return Math.sqrt(this.x * this.x + this.y * this.y);
  }

  // to compute the distance form this point to the given point
  public double distTo(CartPt pt){
    return Math.sqrt((this.x - pt.x) * (this.x - pt.x) +
        (this.y - pt.y) * (this.y - pt.y));
  }
  
  // produce a point moved by the given dx, dy distances
  public CartPt moveXY(int dx, int dy){
    return new CartPt(this.x + dx, this.y + dy);
  }
  
  public CartPt moveRandom(){
    return new CartPt(this.x + 1 - rand.nextInt(3), this.y + 1 - rand.nextInt(3));
  }
}
