/*
 * Book BST
 */

import tester.*;

/*
   +------------------------------+
+->| ICompBook                    |
|  +------------------------------+
|  | int compareBooks(Book, Book) |
|  +------------------------------+
|  
|                  +----------------------+-+
|                  |                      | |
|                  v                      | |                                      
|         +----------------+              | |                
|         |  ABSTBook      |              | |
|         +----------------+              | |
+---------| ICompBook comp |              | |
          +----------------+              | |
                   |                      | |
                   |                      | |
                  /_\                     | |                  |                      | |
           +--------------+               | |
           v              v               | |
  +----------------+   +----------------+ | |    
  | LeafBook       |   | NodeBook       | | |  
  +----------------+   +----------------+ | | 
                       | Book data      | | | 
                     +-| ABSTBook left  |-+ |
                     | | ABSTBook right |---+
                     | +----------------+
                     v
        +---------------+
        | Book          |
        +---------------+
        | String title  |
        | String author |
        | int year      |
        +---------------+  
 */

//to represent a book
class Book{
  String title;
  String author;
  int year;

  Book(String title, String author, int year){
    this.title = title;
    this.author = author;
    this.year = year;
  }

  /* TEMPLATE:
FIELDS:
 ... this.title ...             -- String
 ... this.author ...            -- String
 ... this.year ...              -- int

 METHODS FOR FIELDS:
 ... this.title.equals(String) ...   -- boolean  
 ... this.author.equals(String) ...  -- boolean  

 METHODS FOR THIS CLASS:
   */
}   

interface ICompBook{
  // compare one book to the other one
  public int compareBooks(Book one, Book other);
}

// compare two books by their year of publication
class EarlierYear implements ICompBook{
  public int compareBooks(Book one, Book other){
    return one.year - other.year;
  }
}

// to represent a binary search tree of books
abstract class ABSTBook{
  ICompBook comp;
  
  // compute the total size of this binary search tree
  abstract int size();

  // add the given book to this binary search tree
  abstract ABSTBook add(Book a);
}                                

// to represent a leaf in a binary search tree of books
class LeafBook extends ABSTBook{
  LeafBook(ICompBook comp){
    this.comp = comp;
  }

  // compute the total size of this binary search tree
  public int size(){
    return 0;
  }

  // add the given book to this binary search tree
  public ABSTBook add(Book b){
    return new NodeBook(b, this, this);
  }
}                                              

// to represent a node in a binary search tree of books
class NodeBook extends ABSTBook{
  Book data;
  ABSTBook left;
  ABSTBook right;

  NodeBook(Book data, ABSTBook left, ABSTBook right){
    this.data = data;
    this.left = left;
    this.right = right;
    this.comp = this.left.comp;
  }


  /* TEMPLATE:
   FIELDS:
    ... this.comp ...               -- ICompBook
    ... this.data ...               -- Book
    ... this.left ...               -- ABSTBook
    ... this.right ...              -- ABSTBook

    METHODS FOR FIELDS:
    ... this.comp.compareBooks(Book, Book) ... -- int

    ... this.left.size() ...          -- int
    ... this.left.add(Book) ...       -- ABSTBook

    ... this.right.size() ...         -- int
    ... this.right.add(Book) ...      -- ABSTBook

   */

  // compute the total size of this  binary search tree
  public int size(){
    return 1 + this.left.size() + this.right.size();
  }

  // add the given book to this binary search tree
  public ABSTBook add(Book b){
    if (this.comp.compareBooks(this.data,b) < 0){
      return new NodeBook(this.data,
          this.left,
          this.right.add(b));
    } else {
      return new NodeBook(this.data,
          this.left.add(b),
          this.right);
    }
  }
}                                                    

// Examples and tests for books and BSTs of books
class ExamplesBookBST{
  ExamplesBookBST(){}


  Book cha = new Book("Change in Altitude", "AD", 2009);
  Book oms = new Book("Old Man and the Sea", "EH", 1950);
  Book sar = new Book("Sun Also Rises", "EH", 1954);
  Book htdp = new Book("HtDP", "FFFK", 2000);
    
  ICompBook compYear = new EarlierYear();

  ABSTBook leaf = new LeafBook(this.compYear);
  
  ABSTBook node = new NodeBook(this.sar, this.leaf, this.leaf);
  
  ABSTBook nodes2 = new NodeBook(this.sar, 
                                 new NodeBook(this.oms, this.leaf, this.leaf), 
                                 this.leaf);
  ABSTBook tree = new NodeBook(this.sar, 
                               new NodeBook(this.oms, this.leaf, this.leaf),
                               new NodeBook(this.cha, this.leaf, this.leaf));
                               
  // test the method compare in the EarlierYear class
  void testICompBook(Tester t){
    t.checkExpect(this.compYear.compareBooks(this.oms, this.cha) < 0, true);
    t.checkExpect(this.compYear.compareBooks(this.cha, this.oms) > 0, true);
  }

  // test the method size in the classes that implement IBSTBook
  void testSize(Tester t){
    t.checkExpect(this.leaf.size(),  0);
    t.checkExpect(this.node.size(),  1);
    t.checkExpect(this.tree.size(),  3);
  }

  // test the method add in the classes that implement IBSTBook
  void testAdd(Tester t){
    t.checkExpect(this.leaf.add(this.sar),  this.node);
    t.checkExpect(this.node.add(this.oms),  this.nodes2);
    t.checkExpect(this.node.add(this.cha).add(this.oms),  this.tree);
  }

}