/*
 * Bank accounts BST
 */

import tester.*;

/*
   +------------------------------+
+->| ICompAcct                    |
|  +------------------------------+
|  | int compareAccts(Acct, Acct) |
|  +------------------------------+
|  
|                  +----------------------+-+
|                  |                      | |
|                  v                      | |                                      
|         +----------------+              | |                
|         |  ABSTAcct      |              | |
|         +----------------+              | |
+---------| ICompAcct comp |              | |
          +----------------+              | |
                   |                      | |
                   |                      | |
                  /_\                     | |                  |                      | |
           +--------------+               | |
           v              v               | |
| +----------------+   +----------------+ | |    
| | LeafAcct       |   | NodeAcct       | | |  
| +----------------+   +----------------+ | | 
+-| ICompAcct comp |   | Acct data      | | | 
  +----------------+ +-| ABSTAcct left  |-+ |
                     | | ABSTAcct right |---+
                     | +----------------+
                     v
        +---------------+
        | Acct          |
        +---------------+
        | String owner  |
        | String kind   |
        | int balance   |
        +---------------+  
 */

//to represent an account in a bank
class Acct{
  String owner;
  String kind;
  int balance;

  Acct(String owner, String kind, int balance){
    this.owner = owner;
    this.kind = kind;
    this.balance = balance;
  }

  /* TEMPLATE:
FIELDS:
 ... this.owner ...             -- String
 ... this.kind ...              -- String
 ... this.balance ...           -- int

 METHODS FOR FIELDS:
 ... this.owner.equals(String) ... -- boolean  
 ... this.kind.equals(String) ...  -- boolean  

 METHODS FOR THIS CLASS:
   */
}   

interface ICompAcct{
  // compare one account to the other one
  public int compareAccts(Acct one, Acct other);
}

// compare two accounts by their balances
class LowerBalance implements ICompAcct{
  public int compareAccts(Acct one, Acct other){
    return one.balance - other.balance;
  }
}

// to represent a binary search tree of accounts
abstract class ABSTAcct{
  ICompAcct comp;
  
  // compute the total size of this binary search tree
  abstract int size();

  // add the given account to this binary search tree
  abstract ABSTAcct add(Acct a);
}                                

// to represent a leaf in a binary search tree of accounts
class LeafAcct extends ABSTAcct{
  LeafAcct(ICompAcct comp){
    this.comp = comp;
  }

  // compute the total size of this binary search tree
  public int size(){
    return 0;
  }

  // add the given account to this binary search tree
  public ABSTAcct add(Acct b){
    return new NodeAcct(b, this, this);
  }
}                                              

// to represent a node in a binary search tree of accounts
class NodeAcct extends ABSTAcct{
  Acct data;
  ABSTAcct left;
  ABSTAcct right;

  NodeAcct(Acct data, ABSTAcct left, ABSTAcct right){
    this.data = data;
    this.left = left;
    this.right = right;
    this.comp = this.left.comp;
  }


  /* TEMPLATE:
   FIELDS:
    ... this.comp ...               -- ICompAcct
    ... this.data ...               -- Acct
    ... this.left ...               -- ABSTAcct
    ... this.right ...              -- ABSTAcct

    METHODS FOR FIELDS:
    ... this.comp.compareAccts(Acct, Acct) ... -- int

    ... this.left.size() ...          -- int
    ... this.left.add(Acct) ...       -- ABSTAcct

    ... this.right.size() ...         -- int
    ... this.right.add(Acct) ...      -- ABSTAcct

   */

  // compute the total size of this binary search tree
  public int size(){
    return 1 + this.left.size() + this.right.size();
  }

  // add the given account to this binary search tree
  public ABSTAcct add(Acct b){
    if (this.comp.compareAccts(this.data,b) < 0){
      return new NodeAcct(this.data,
          this.left,
          this.right.add(b));
    } else {
      return new NodeAcct(this.data,
          this.left.add(b),
          this.right);
    }
  }
}                                                    

// Examples and tests for accounts and BSTs of accounts
class ExamplesAcctBST{
  ExamplesAcctBST(){}


  Acct chk1 = new Acct("JFK", "checking", 4000);
  Acct chk2 = new Acct("LBJ", "checking", 1000);
  Acct sav = new Acct("DDE", "savings", 2000);
  Acct credit = new Acct("RMN", "credit card", 3000);
    
  ICompAcct compBalance = new LowerBalance();

  ABSTAcct leaf = new LeafAcct(this.compBalance);
  
  ABSTAcct node = new NodeAcct(this.sav, this.leaf, this.leaf);
  
  ABSTAcct nodes2 = new NodeAcct(this.sav, 
                                 new NodeAcct(this.chk2, this.leaf, this.leaf), 
                                 this.leaf);
  ABSTAcct tree = new NodeAcct(this.sav, 
                               new NodeAcct(this.chk2, this.leaf, this.leaf),
                               new NodeAcct(this.chk1, this.leaf, this.leaf));
                               
  // test the method compare in the LowerBalance class
  void testICompAcct(Tester t){
    t.checkExpect(this.compBalance.compareAccts(this.chk2, this.chk1) < 0, true);
    t.checkExpect(this.compBalance.compareAccts(this.chk1, this.chk2) > 0, true);
  }

  // test the method size in the classes that implement IBSTAcct
  void testSize(Tester t){
    t.checkExpect(this.leaf.size(),  0);
    t.checkExpect(this.node.size(),  1);
    t.checkExpect(this.tree.size(),  3);
  }

  // test the method add in the classes that implement IBSTAcct
  void testAdd(Tester t){
    t.checkExpect(this.leaf.add(this.sav),  this.node);
    t.checkExpect(this.node.add(this.chk2),  this.nodes2);
    t.checkExpect(this.node.add(this.chk1).add(this.chk2),  this.tree);
  }

}