import world.*;
import image.*;

/**
 * To represent a city in the USA
 */
class City{
  String zip;
  String name;
  String state;
  Loc loc;
  
  /**
   * The standard full constructor
   * @param zip the given zip code
   * @param name the name of this city
   * @param state the state for this city
   * @param loc the location data
   */
  City(String zip, String name, String state, Loc loc){
    this.zip = zip;
    this.name = name;
    this.state = state;
    this.loc = loc;
  }
  
  /**
   * The constructor with latitude and longitude given directly
   * @param zip the given zip code
   * @param name the name of this city
   * @param state the state for this city
   * @param longitude the longitude for this city
   * @param latitude the latitude for this city
   */  
  City(String zip, String name, String state, 
        int longitude, int latitude){
    this.zip = zip;
    this.name = name;
    this.state = state;
    this.loc = new Loc(longitude, latitude);
  }
  
  /* TEMPLATE:
  FIELDS
   ... this.name ...       -- String
   ... this.state ...      -- String
   ... this.loc ...        -- Loc
   ... this.zip ...        -- String
 
  METHODS
   ... this.distanceTo(City) ...       -- double
   ... this.drawCity(Scene) ...        -- Scene
 
  METHODS FOR FIELDS
   ... this.loc.distanceTo(Loc) ...       -- double
   ... this.loc.toPosn() ...              -- Posn
  */  
  
  /**
   * Compute the distance from this city to the given city
   * 
   * @param that the given city
   */
  double distanceTo(City that){
    return this.loc.distanceTo(that.loc);
  }
  
  /**
   * Produce the <code>Posn</code> for the location of this city 
   * on a 400 x 400 Scene
   * @return the desired location
   */
  Posn toPosn(){
    return this.loc.toPosn();
  }
  
  /**
   * Draw this city on the given <code>Scene<code> 
   * (assumed to be 400 x 400)
   * @param s the given <code>Scene</code>
   * @return the scene with this city shown
   */
  Scene drawCity(Scene s){
    return s.placeImage(new Circle(3, "solid", "red"), this.toPosn());
  }
  
  public String toString(){
    return this.name + ", " + this.state + ", " + this.zip +
           " (" + this.loc.toString() + ")"; 
  }
}

/**
 * To represent a location on a map
 */
class Loc{
  int longitude;
  int latitude;
  
  /**
   * The full constructor
   * @param longitude the give longitude
   * @param latitude the given latitude
   */
  Loc(int longitude, int latitude){
    this.longitude = longitude;
    this.latitude = latitude;
  }
  
  /* TEMPLATE:
  FIELDS
   ... this.longitude ...     -- int
   ... this.latitude ...      -- int
 
  METHODS
   ... this.distanceTo(Loc3) ...      -- double
   ... this.toX() ...                 -- int
   ... this.toY() ...                 -- int
   ... this.toPosn() ...              -- Posn
  */  
  
  /**
   * Compute the distance from this location to the 
   * given location
   * 
   * @param that the given <code>Loc</code>
   * @return the distance between the two locations
   */
  double distanceTo(Loc that){
    return Math.sqrt(55 * (this.latitude - that.latitude) * 
                     55 *  (this.latitude - that.latitude) +
                     70 *  (this.longitude - that.longitude) * 
                     70 * (this.longitude - that.longitude));
  }
  
  /**
   * Convert the longitude to the x coordinate in the 400 by 400 scene
   * @param lng the given longitude
   * @return the desired x coordinate
   */
  private int toX(int lng){
    return 400 - 400 * (lng - 65) / 60;
  } 
  
  /**
   * Convert the latitude to the x coordinate in the 400 by 400 scene
   * @param lat the given latitude
   * @return the desired x coordinate
   */
  private int toY(int lat){
    return 400 - 400 * (lat - 20) / 30;
  }
  
  /**
   * Represent this location as a <code>Posn</code> in a 
   * 400 by 400 scene
   * @return the desired <code>Posn</code>
   */
  Posn toPosn(){
    return new Posn(this.toX(this.longitude), 
                    this.toY(this.latitude));
  }  
  
  public String toString(){
    return this.longitude + ", " + this.latitude;
  }
}