/** 
 * Algorithms class 
 * @author Bryan Chadwick
 * */

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Random;


// The classic Algorithms class for static method implementations
public class Algorithms {
    // The swap we've seen in lab and in class (and in HW)
    static <X> void swap(ArrayList<X> list, int i, int j){
        list.set(i, list.set(j, list.get(i)));
    }
    
    // Basic insertion sort algorithm
    // Starting from the back, insert the current element into the 
    //   sorted 'rest' of the list. The last element is trivially sorted  
    static <X> void insertionSort(ArrayList<X> list, Comparator<X> comp){
        for(int i = list.size()-2; i >= 0; i--)
            insert(list, i, comp);
    }
    
    // Insert as described in clas... swap the given element into place
    //   in a sorted list starting at index (i+1)
    static <X> void insert(ArrayList<X> list, int i, Comparator<X> comp){
        while(i < list.size()-1 && comp.compare(list.get(i+1), list.get(i)) < 0){
            swap(list, i, i+1);
            i++;
        }
    }

    // Quick Sort algorithm... only has to do with partitioning a section of
    //   the ArrayList into lessThan and greaterThan parts then recuring. We
    //   assume that a random pick will give us a decent pivot element.
    static <X> void quickSort(ArrayList<X> list, Comparator<X> comp){
        partition(list, comp, 0, list.size());
    }

    /* 
       Partition elements [start..end) of the list based on a randomly chosen
         pivot point. We keep track of to 'regions', the lessThan, and greaterThan
         regions are created, then we swap the pivot to it's right place in the
         ArrayList.  Note that the 'end' is exclusive, not inclusive.
    */
    static <X> void partition(ArrayList<X> list, Comparator<X> comp,
                              int start, int end){
        // Only one element... it's already sorted
        if(start >= end)return;

        // Random pivot, and starting indexes for the regions
        int pivot = randInt(start, end),
            less = start+1,
            greater = less;

        // Move the pivot to the front of the ArrayList
        swap(list, start, pivot);

        // Cache the pivot...
        X target = list.get(start);

        // While the regions are not the whole ArrayList
        while(greater < end){
            // If the current element is less than the pivot then
            //   it belongs in the 'left' region.  So we may need
            //   to swap it there to mantain the regions
            if(comp.compare(list.get(greater), target) < 0){
                if(less != greater)
                    swap(list, less, greater);
                // The 'lessThan' region has gotten bigger
                less++;
            }
            // The greater region always moves back since the lessThan
            //   region is in front of it.
            greater++;
        }
        // Swap the pivot back into a good spot... maintaining our regions
        swap(list, start, less-1);
        partition(list, comp, start, less-1);
        partition(list, comp, less, end);
    }

    // Return a nice random integer between: [low, high)
    static Integer randInt(int low, int high){ 
        return new Random().nextInt(high-low)+low; 
    }   
    
    // 
    static <T> AList<T> insertionSortList(AList<T> alist, Comparator<T> comp){
      return alist.isort(comp);
    }
}


abstract class AList<T>{
  abstract AList<T> isort(Comparator<T> comp);
  abstract AList<T> insert(T t, Comparator<T> comp);
}

class MT<T> extends AList<T>{
  AList<T> isort(Comparator<T> comp){ return this; }
  
  AList<T> insert(T t, Comparator<T> comp){
    return new Cons<T>(t, this);
  }
}
  
class Cons<T> extends AList<T>{
  T first;
  AList<T> rest;
  
  Cons(T first, AList<T> rest){
    this.first = first;
    this.rest = rest;
  }
  
  AList<T> isort(Comparator<T> comp){ 
    return this.rest.isort(comp).insert(this.first, comp); 
  }
  
  AList<T> insert(T t, Comparator<T> comp){
    if (comp.compare(this.first, t) < 0)
      return new Cons<T>(this.first, this.rest.insert(t, comp));
    else 
      return new Cons<T>(t, this);
  }
}


