/*
 * @(#)SimplePainter.java  1.0  14 July 2001
 *
 * Copyright 2001
 * College of Computer Science
 * Northeastern University
 * Boston, MA  02115
 *
 * This software may be used for educational purposes as long as
 * this copyright notice is retained intact at the top of all files.
 *
 * Should this software be modified, the words "Modified from 
 * Original" must be included as a comment below this notice.
 *
 * All publication rights are retained.  This software or its 
 * documentation may not be published in any media either in whole
 * or in part without explicit permission.
 *
 * Contact information:
 *   Richard Rasala    rasala@ccs.neu.edu
 *   Viera Proulx      vkp@ccs.neu.edu
 *   Jeff Raab         jmr@ccs.neu.edu
 * 
 * Telephone:          617-373-2462
 *
 * This software was created with support from Northeastern 
 * University and from NSF grant DUE-9950829.
 */

import edu.neu.ccs.*;
import edu.neu.ccs.gui.*;
import edu.neu.ccs.util.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import javax.swing.*;

/** A simple example of using the mouse adapter in the graphics buffered panel
 *      The TextFieldView below mirrors the current mouse position
 *      On mouse click, the location is 'captured', 
 *          and a line is drawn to that spot
 *      ColorView is used to select the color for the line to be drawn
 *      The History of the painting is recorded in a TextAreaView
 * @author  Viera K. Proulx 
 * @author  Jennifer McDonald
 * @author  Richard Rasala
 * @version 14 July 2001
 */

public class SimplePainter extends DisplayPanel
    implements JPTConstants {
    
    
    /////////////////
    // Static Data //
    /////////////////
    
    // constant: buffer size
    private static final int bufferSize = 400;
    
    // constants: settings for the default first line
    private static final String X = "100";
    private static final String Y = "200";
    private static final String Mx = "300";
    private static final String My = "100";
    
   	
    /////////////////
    // Member Data //
    /////////////////

    ////////////////////
    // State variable //
    ////////////////////
    
    // if true, no line is drawn from the previous point
    private boolean starting = true;
    
        
    /////////////////////////////////
    // Input text and color fields //
    /////////////////////////////////
        
    // fields that record the selected point    
    private TextFieldView xTFV =
        new TextFieldView(
            "200",                      // initial value shown in the TFV
            "Fix X:",                   // useless error message - value will
            "X Coordinate Error");      // always come from xMouseTFV
    
    private TextFieldView yTFV =
        new TextFieldView(
            "200",                      // initial value shown in the TFV
            "Fix Y:",                   // useless error message - value will
            "Y Coordinate Error");      // always come from yMouseTFV


    // fields that record the current mouse position
    private TextFieldView xMouseTFV =
        new TextFieldView(
            "0",                        // initial value shown in the TFV
            "Fix X:",                   // useless error message - value will
            "X Coordinate Error");      // always come from MouseListener
    
    private TextFieldView yMouseTFV =
        new TextFieldView(
            "0",                        // initial value shown in the TFV
            "Fix Y:",                   // useless error message - value will
            "Y Coordinate Error");      // always come from MouseListener
   
    // color view where user can select the color for the drawing
    private ColorView color =
        new ColorView(Color.red);  // initial color is red
        
    
    ////////////////////////////////////
    // Output text area and file view //
    ////////////////////////////////////
   
    // panel to display past graphed points
    private TextAreaView history =
        new TextAreaView();
        

    /////////////
    // Actions //
    /////////////    

    // the actions for the actions panel
    
    /*
     * the clear action is constructed using
     *   the String "Clear" to name its button
     *   and specifying the clear method to be performed when pressed
     */

    private SimpleAction clear = 
        new SimpleAction("Clear") {
           public void perform(){ clear(); }
        };
    
    
    /*
     * the startNewPolygon action is constructed using
     *   the String "New Polygon" to name its button
     *   and specifying the startNewPolygon method to be performed when pressed
     */

    private SimpleAction startNewPolygon = 
        new SimpleAction("New Polygon") {
           public void perform(){ startNewPolygon(); }
        };
    
        
    // list of actions to be included in the actions panel
    /* 
     * array of actions is constructed
     */
    private Action[] actionList = {clear, startNewPolygon};

    // actions panel
    /*
     * actions panel is constructed
     * with the actions it will contain supplied as parameter
     */ 
    private ActionsPanel actions = new ActionsPanel(actionList);

    


    /////////////////
    // GUI Section //
    /////////////////
    
    // overall control panel for the GUI
    private DisplayCollection controls =
        new DisplayCollection();

    // overall control panel for the start point information
    private DisplayCollection currentPoint =
        new DisplayCollection();

    // overall control panel for the end point information
    private DisplayCollection mousePoint =
        new DisplayCollection();
        
    // control panel to combine point data displays
    private DisplayCollection pointData = 
        new DisplayCollection(HORIZONTAL);

    // square window for painting
    private BufferedPanel window =
        new BufferedPanel(bufferSize, bufferSize);
    
    // graphics context to draw lines and other shapes
    private Graphics2D G  = null;
        

    //////////////////        
    // Main Program //
    //////////////////
        
    /* 
     * create a window
     *   titled "Simple Line Drawing"
     *   whose contents are defined by the SimpleDrawing constructor
     */
    public static void main(String[] args) {
        
        JPTFrame.createQuickJPTFrame("Simple Painter", new SimplePainter());            
    }
    
    
    /////////////////
    // Constructor //
    /////////////////
    
   /*** Constructor for SimplePainter program:
    **** creates an action panel with two actions (clear and New Polygon)
    **** adds two input views for endpoints of a line
    **** adds two output views to display the current x and y mouse position
    **** sets sizes for all four views and suggestions for the input views
    **** adds a control group and a buffered panel window for drawing
    **** accesses the mouseActionAdapter and defines mouseClicked actions
    ***/

    public SimplePainter() {
        // layout for panel as a whole
        setLayout(new BorderLayout());


        //////////////////
        // View section //
        //////////////////
        
        // set preferred width for the text fields
        xTFV.setPreferredWidth(150);
        yTFV.setPreferredWidth(150);
        xMouseTFV.setPreferredWidth(150);
        yMouseTFV.setPreferredWidth(150);
        
        // set the default view states for the text fields
        xTFV.setDefaultViewState(X);
        yTFV.setDefaultViewState(Y);
        xMouseTFV.setDefaultViewState(Mx);
        yMouseTFV.setDefaultViewState(My);

        // set preferred size for the history text area
        history.setColumns(13);
        history.setRows(18);
        
        
        /////////////////
        // GUI Section //
        /////////////////
        
        // activate color chooser using single click
        color.setChooserClickCount(1);
        
        ////////////////////////////////////////////
        // add text fields to the point data Display

        // add two text field views to build start point display
        currentPoint.add(
            new Display(xTFV, "X:", null));
        
        currentPoint.add(
            new Display(yTFV, "Y:", null));
        
        // add two text field views to build end point display
        mousePoint.add(
            new Display(xMouseTFV, "X:", null));
        
        mousePoint.add(
            new Display(yMouseTFV, "Y:", null));
                
        // add the start and end point displays to the point data display 
        pointData.add(
            new Display(currentPoint, null, "Last end point"));
            
        pointData.add(
            new Display(mousePoint, null, "Current mouse location"));
            
        ///////////////////////////////////////////////////
        // add color view, actions panel, and history panel
        // to the controls DisplayCollection

        // add color view to controls
        controls.add(
            new DisplayWrapper(
               new Display(color, "Color:", null)));
           
        // add actions panel to controls  
        controls.add(actions);
        
        // install history panel in a scrollable display and 
        // add to controls wrapped with a title
        controls.add(
            new DisplayWrapper(
                new Display(new ScrollableDisplay(history), null, "Lines drawn")));
                
                
        ///////////////////
        // Mouse Section //
        ///////////////////

        // get the mouse action adapter from the graphics window panel
        MouseActionAdapter adapter = window.getMouseActionAdapter();
        
        // draw if mouse clicked
        adapter.addMouseClickedAction(new MouseAction() {
           public void mouseActionPerformed(MouseEvent mevt) {
             draw(mevt);
           }
         });
                    
        // track mouse motions
        adapter.addMouseMovedAction(new MouseAction() {
           public void mouseActionPerformed(MouseEvent mevt) {
            track(mevt);
           }        
         });            
                           
                 
        /////////////////////////////////////////////////
        // wrap the graphics window into a titled display
        Display windowDisplay =
            new Display(window, null, "Graphics");

        ////////////////////////////////////////////////////////////////
        // add controls, the point data display, and the graphics window 
        // to the main panel
        add(controls, BorderLayout.WEST);
        add(windowDisplay, BorderLayout.CENTER);
        add(pointData, BorderLayout.SOUTH);


        // clear window
        reset();
        setEnabled(true);
    }
 
    
    ////////////////////////
    // Action Definitions //
    ////////////////////////  
    
    /*
     * clear action
     * 
     * reset the text fields to the default initial default values
     * reset the color choice to red
     * clear the history text area
     * erase the graphics window
     */
    public void clear() {
    
        // reset all components
        reset();
        
        // clear the graphics window
        window.clearPanel();
        repaint();
        
        // reset the starting state variable to indicate start of a new polygon            
        starting = true;
    }
    
 
    /*
     * startNewPolygon action
     *
     * reset the starting state variable to indicate start of a new polygon
     */
    
    public void startNewPolygon() {
        starting = true;
    }
    

    //////////////////////////////
    // Mouse Action Definitions //
    //////////////////////////////
    
    /**
     * Mouse moved action for tracking the position of mouse 
     *  in the graph window
     */
     
    public void track(MouseEvent mevt) {
        
        // record the mouse position in its text field view
        xMouseTFV.setViewState((int)mevt.getX() + "");
        yMouseTFV.setViewState((int)mevt.getY() + "");
         
   }
	

    /**
     * Mouse clicked action for finding position of mouse in the graph window
     *  and drawing the line to this position (if not a new polygon)
     */
    public void draw(MouseEvent mevt) {

        // get the graphics context to draw the line
        Graphics2D G = window.getBufferGraphics();
        
        // Line2D object to draw
        Line2D.Double  L = new Line2D.Double();
        
        // get the start point from text field view
        Point2D.Double P1 = 
            new Point2D.Double(xTFV.demandDouble(), yTFV.demandDouble());
        
        // get the end point from mouse location
        Point2D.Double P2 = 
            new Point2D.Double(mevt.getX(), mevt.getY());
                
        // if not a new polygon, draw a line
        if (!starting){
        
            // set paint color to user color choice
            G.setPaint(color.getColor());

            // draw the line from P1 to P2
                L.setLine (P1, P2);
                G.draw(L);
                
            // show the color and the line coordinates for history
            history.append("(" + color.getViewState() + ")\n");
            history.append("(" + (int)P1.x + ", " + (int)P1.y + ") : ");
            history.append("(" + (int)P2.x + ", " + (int)P2.y + ")\n");
                
            window.repaint();
       }
        
       // set the start point text field to current end point
       xTFV.setViewState((int)P2.x + "");
       yTFV.setViewState((int)P2.y + "");
        
       // indicate polygon was started
       starting = false;

    }
    
        
}
