/*
 * @(#)SimpleDrawing.java  1.0  14 July 2001
 *
 * Copyright 2001
 * College of Computer Science
 * Northeastern University
 * Boston, MA  02115
 *
 * This software may be used for educational purposes as long as
 * this copyright notice is retained intact at the top of all files.
 *
 * Should this software be modified, the words "Modified from 
 * Original" must be included as a comment below this notice.
 *
 * All publication rights are retained.  This software or its 
 * documentation may not be published in any media either in whole
 * or in part without explicit permission.
 *
 * Contact information:
 *   Richard Rasala    rasala@ccs.neu.edu
 *   Viera Proulx      vkp@ccs.neu.edu
 *   Jeff Raab         jmr@ccs.neu.edu
 * 
 * Telephone:          617-373-2462
 *
 * This software was created with support from Northeastern 
 * University and from NSF grant DUE-9950829.
 */

import edu.neu.ccs.*;
import edu.neu.ccs.gui.*;
import edu.neu.ccs.util.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import javax.swing.*;

/**
 * A simple line drawing program
 *   Two pairs of TextFieldViews let user select the start and end of a line
 *   ColorView is used to select the color for the line to be drawn
 *   After a line is drawn, the coordinates of the end point are copied
 *   into the TextFieldView of the start point 
 *   to allow for a dot-to-dot drawing
 *   The History TextAreaView displays the coordinates of lines drawn
 *
 * @author  Viera K. Proulx 
 * @author  Jennifer McDonald
 * @author  Richard Rasala
 * @version 14 July 2001
 */
public class SimpleDrawing extends DisplayPanel
    implements JPTConstants {
    
    /////////////////
    // Static Data //
    /////////////////
    
    // constant: buffer size for the graphics display
    private static final int bufferWidth = 400;
    private static final int bufferHeight = 400;
    
    // constants: settings for the default first line
    private static final String X1 = "100";
    private static final String Y1 = "200";
    private static final String X2 = "300";
    private static final String Y2 = "100";
        

    /////////////////
    // Member Data //
    /////////////////

    ///////////////////////
    // Input text fields //
    ///////////////////////
        
    // fields that record the first selected point    
    private TextFieldView x1TFV =
        new TextFieldView(
            X1,                          // initial value displayed in the TFV
            "Fix X1:",                   // prompt for correcting the input
            "X1 Coordinate Error");      // title for the error dialog box
    
    private TextFieldView y1TFV =
        new TextFieldView(
            Y1,                          // initial value displayed in the TFV
            "Fix Y1:",                   // prompt for correcting the input
            "Y1 Coordinate Error");      // title for the error dialog box

    // fields that record the second selected point    
    private TextFieldView x2TFV =
        new TextFieldView(
            X2,                          // initial value shown in the TFV
            "Fix X2:",                   // prompt for correcting the input
            "X2 Coordinate Error");      // title for the error dialog box
    
    private TextFieldView y2TFV =
        new TextFieldView(
            Y2,                          // initial value shown in the TFV
            "Fix Y2:",                   // prompt for correcting the input
            "Y2 Coordinate Error");      // title for the error dialog box
   
    // color view where user can select the color for the drawing
    private ColorView color =
        new ColorView(Color.red, true);  // initial color is red
                                         // include text field for colors
    
    
    /////////////
    // Actions //
    /////////////    

    // the actions for the actions panel
    
    /*
     * the clear action is constructed using
     *   the String "Clear" to name its button
     *   and specifying the clear method to be performed when pressed
     */

    private SimpleAction clear = 
        new SimpleAction("Clear") {
           public void perform(){ clear(); }
        };
    
    /*
     * the draw action is constructed using
     *   the String "Draw" to name its button
     *   and specifying the draw method to be performed when pressed
     */

    private SimpleAction draw = 
        new SimpleAction("Draw") {
           public void perform(){ draw(); }
        };
    
    // list of actions to be included in the actions panel
    /* 
     * array of actions is constructed
     */
    private Action[] actionList = {clear, draw};

    // actions panel
    /*
     * actions panel is constructed
     * with the actions it will contain supplied as parameter
     */ 
    private ActionsPanel actions = new ActionsPanel(actionList);

    
    /////////////////
    // GUI Section //
    /////////////////
    
    // overall control panel for the GUI
    private DisplayCollection controls =
        new DisplayCollection();

    // overall control panel for the start point information
    private DisplayCollection startPoint =
        new DisplayCollection();

    // overall control panel for the end point information
    private DisplayCollection endPoint =
        new DisplayCollection();
        
    // control panel to combine point data displays
    private DisplayCollection pointData = 
        new DisplayCollection(HORIZONTAL);

    // square window for painting
    private BufferedPanel window =
        new BufferedPanel(bufferWidth, bufferHeight);
            
    // panel to display past graphed points
    private TextAreaView history =
        new TextAreaView();

    
    //////////////////        
    // Main Program //
    //////////////////
        
    /* 
     * create a window
     *   titled "Simple Line Drawing"
     *   whose contents are defined by the SimpleDrawing constructor
     */
    public static void main(String[] args) {
        
        JPTFrame.createQuickJPTFrame("Simple Line Drawing", new SimpleDrawing());            
    }
    
    
    /////////////////
    // Constructor //
    /////////////////
    
    /* 
     * constructor for SimpleDrawing program:
     *   installs two action buttons - add and clear - in the action panel 
     *   sets sizes and suggestions for four input views of endpoints of a line
     *
     *   installs color view, actions panel, and history view 
     *      in the controls display
     *   
     *   installs two pairs of input views in point data display
     
     *   wraps and titles the controls, the point data display 
     *   and the buffered panel and installs them in the main GUI panel
     */
    
    public SimpleDrawing() {
        // layout for panel as a whole
        setLayout(new BorderLayout());

        //////////////////
        // View section //
        //////////////////
        
        // set preferred width for the text fields
        x1TFV.setPreferredWidth(120);
        y1TFV.setPreferredWidth(120);
        x2TFV.setPreferredWidth(120);
        y2TFV.setPreferredWidth(120);
        
        // set the default view states for the text fields
        x1TFV.setDefaultViewState(X1);
        y1TFV.setDefaultViewState(Y1);
        x2TFV.setDefaultViewState(X2);
        y2TFV.setDefaultViewState(Y2);
         
        // set preferred size for the history text area
        history.setColumns(13);
        history.setRows(18);
        
        // set suggestions for the first point
        x1TFV.getInputProperties().setSuggestion("100");
        y1TFV.getInputProperties().setSuggestion("100");

        // set suggestions for the second point
        x2TFV.getInputProperties().setSuggestion("200");
        y2TFV.getInputProperties().setSuggestion("200");
        
        
        /////////////////
        // GUI Section //
        /////////////////
        
        // activate color chooser using single click
        color.setChooserClickCount(1);
        
        ////////////////////////////////////////////
        // add text fields to the point data Display

        // add two text field views to build start point display
        startPoint.add(
            new Display(x1TFV, "X1:", null));
        
        startPoint.add(
            new Display(y1TFV, "Y1:", null));
        
        // add two text field views to build end point display
        endPoint.add(
            new Display(x2TFV, "X2:", null));
        
        endPoint.add(
            new Display(y2TFV, "Y2:", null));
                
        // add the start and end point displays to the point data display 
        pointData.add(
            new Display(startPoint, null, "Start point of the line"));
            
        pointData.add(
            new Display(endPoint, null, "End point of the line"));
            
        ///////////////////////////////////////////////////
        // add color view, actions panel, and history panel
        // to the controls DisplayCollection

        // add color view to controls
        controls.add(
            new DisplayWrapper(
               new Display(color, null, "Color")));
           
        // add actions panel to controls  
        controls.add(actions);
        
        // install history panel in a scrollable display and 
        // add to controls wrapped with a title
        controls.add(
            new DisplayWrapper(
                new Display(new ScrollableDisplay(history), null, "Lines drawn")));
 
        /////////////////////////////////////////////////
        // wrap the graphics window into a titled display
        Display windowDisplay =
            new Display(window, null, "Graphics");

        ////////////////////////////////////////////////////////////////
        // add controls, the point data display, and the graphics window 
        // to the main panel
        add(controls, BorderLayout.WEST);
        add(windowDisplay, BorderLayout.CENTER);
        add(pointData, BorderLayout.SOUTH);
    }
    
    
    ////////////////////////
    // Action Definitions //
    ////////////////////////  
    
    /*
     * clear action
     * 
     * reset the text fields to the default initial default values
     * reset the color choice to red
     * clear the history text area
     * erase the graphics window
     */
    public void clear() {
               
        // reset all components
        reset();
        
                // Note: the reset() is called recursively for all components
                //   That means that the following four statements will be invoked:
                //     controls.reset();           which will then call
                //       color.reset();
                //       history.reset();
                //       actions.reset();
                //
                //     pointData.reset();          which will then call
                //       startPoint.reset();
                //       endPoint.reset();
                //
                //   to reset all states to their initial defaults
                 
        // clear the graphics window
        window.clearPanel();
        repaint();
        
    }
    
    /*
     * draw action
     * 
     * draw a line between the two points selected by the user 
     */
    public void draw() {
 
        // get the graphics context to draw the line
        Graphics2D G = window.getBufferGraphics();

        // starting point P1 - set to the coordinates given by the user 
        Point2D.Double P1 = 
            new Point2D.Double( x1TFV.demandDouble(), y1TFV.demandDouble() );
        
        // ending point P2 - set to the coordinates given by the user 
        Point2D.Double P2 = 
            new Point2D.Double( x2TFV.demandDouble(), y2TFV.demandDouble() );
        
        // set paint color to user color choice
        G.setPaint(color.getColor());
 
        // line to draw - from P1 to P2
        Line2D.Double  L = new Line2D.Double(P1, P2);
        G.draw(L);
      
        // show the color and the line coordinates for history
        history.append("(" + color.getViewState() + ")\n");
        history.append("(" + (int)P1.x + ", " + (int)P1.y + ") : ");
        history.append("(" + (int)P2.x + ", " + (int)P2.y + ")\n");
        
        // set the start point to the coordinates of the previous end point
        x1TFV.setViewState((int)P2.x + "");
        y1TFV.setViewState((int)P2.y + "");
        
        // set the end point to a new random point within the graphics window
        int x = (int) (bufferWidth  * Math.random());
        int y = (int) (bufferHeight * Math.random());
        x2TFV.setViewState(x + "");
        y2TFV.setViewState(y + "");
        
        repaint();
    }
}
