/*
 * @(#)ZipCode.java  1.0  16 April 2003
 *
 * @author Viera K. Proulx
 */

import edu.neu.ccs.gui.*;   // to get BufferedPanel
import edu.neu.ccs.*;	    // to get XInt
import java.util.*;         // to get StringTokenizer
import java.text.*;         // to get ParseException

/** Represents a zip code: includes city, location, and population */
class ZipCode implements Stringable {
    
    /*-------------------------------------------------------------------------
     * Member data 
     *-----------------------------------------------------------------------*/
    
    /** The zip code */
    int zip;
    
    /** The City to which this zip code belongs */
    City city;
    
    /** The longitude of the zip code, in degrees west */
    double longitude;
    
    /** The latitude of the zip code, in degrees north */
    double latitude;

    /** The population of this zip code */
    int population;

    /** Decimal format to print leading zeros in zip code */
    static final DecimalFormat zipFormat = new DecimalFormat("00000");
    
    /*-------------------------------------------------------------------------
     * Constructor
     *-----------------------------------------------------------------------*/

    /* full constructor */
    public ZipCode(int zip, City city, double longitude, double latitude,
                   int population) {
        this.zip = zip;
        this.city = city;
        this.longitude = longitude;
        this.latitude = latitude;
        this.population = population;
    }
    
    /* convenience constructor */
    public ZipCode(int zip, String cityname, String state, 
                   double longitude, double latitude, int population){
        this(zip, new City(cityname, state), longitude, latitude, population);
    }
    
    /* copy constructor */
    public ZipCode(ZipCode zipcode){
        this(zipcode.zip, zipcode.city, zipcode.longitude, zipcode.latitude,
             zipcode.population);
    }
    
    /* constructor which parses string into zip code */
    public ZipCode(String zip, String name, String state, 
                   double longitude, double latitude, int population) {
        this(decimal(zip), name, state, longitude, latitude, population);
    }

    /* default constructor */
    public ZipCode() {
        this(0, null, 0.0, 0.0, 0);
    }

    /*-------------------------------------------------------------------------
     * Methods 
     *-----------------------------------------------------------------------*/
        
    /** determine whether this zip code is extensionally equal to 
     * the given object */
    public boolean equals(Object obj) {
	if (obj instanceof ZipCode) {
	    ZipCode zc = (ZipCode) obj;
	    return this.zip == zc.zip &&
		this.city.equals(zc.city) &&
		this.longitude == zc.longitude &&
		this.latitude == zc.latitude &&
                this.population == zc.population;
	} else {
	    return false;
	}
    }
    
    /*-------------------------------------------------------------------------
     * Stringable interface
     *-----------------------------------------------------------------------*/
    /** city data to string for file output*/
    public String toStringData(){
    	String s = zipFormat.format(this.zip) + ", " +
	    this.city.name  + ", " +
	    this.city.state + ", " +
	    this.longitude + ", " + 
	    this.latitude + ", " + 
            this.population + "\n";
    	return s;
    }
    
    /** string to zip code data */
    public void fromStringData(String s) throws java.text.ParseException {
        try {
            /** set up the string tokenizer to extract zip code data */
            StringTokenizer st = new StringTokenizer(s, ",");

            this.zip   = decimal(st.nextToken());
            String city = st.nextToken();
            String state = st.nextToken();
            this.city = new City(city, state);
	    this.longitude = doubleValue(st.nextToken());
            this.latitude  = doubleValue(st.nextToken());
            this.population = intValue(st.nextToken());
        } catch(NoSuchElementException e){
            System.out.println(e.getMessage());
            throw new java.text.ParseException("Tokenizer Error", 0);
        }
    }

    /** convert a String to int - if possible */
    protected static int intValue(String s) throws ParseException{
        return (new XInt(s)).intValue();
    }

    protected static double doubleValue(String s) {
	try {
	    return Double.parseDouble(s);
	} catch (NumberFormatException nfe) {
	    throw new RuntimeException("bad double format");
	}
    }

    /*-------------------------------------------------------------------------
     * Console print
     *-----------------------------------------------------------------------*/
    /** print the zip code data */
    public String toString(){
    	return ("new " + getClass() + "(" + 
    	        zipFormat.format(this.zip)   + ", " + 
    	        this.city.toString() + ", " + 
    	        this.longitude + ", " + 
		this.latitude + ", " + 
                this.population + ")\n");
    }
    
    /** helper method to convert zip code string to integer */
    public static int decimal(String s){
	
    	char[] c = s.toCharArray();
    	int n = 0;
    	try {
	    for (int i = 0; i < 5; i++){
    		n = 10 * n + intValue("" + c[i]);
	    }
    	}
    	catch(ParseException e){
	    System.out.println("Converting zip to int " + e);
    	}
    	return n;
    }
}
