/* ------------------------------------------------------------------
   Functional (External) Iterator Pattern:
        iterator for a list of balloons 
*/

import edu.neu.ccs.*;
import edu.neu.ccs.gui.*;
import edu.neu.ccs.util.*;

class InGuiRange implements IRange {

    /*-------------------------------------------------------- 
     Member data
     *------------------------------------------------------*/
    /* a balloon object to hold values */
    protected Balloon b = new Balloon();
    
    /* determines whether the input dialog has been closed */
    protected boolean closed = true;
    
    /* determines whether new baloon data has been submitted */
    protected boolean submitted = false;
    
    
    /*-------------------------------------------------------- 
     Member data for the GUI
     *------------------------------------------------------*/
    /* the view for one balloon */
    protected BalloonView bView; 
    
    /* the general dialog for balloon input */
    protected GeneralDialog input;
        
    /** The submit action. */
    protected SimpleAction submitAction =
        new SimpleAction("Submit") {
            public void perform() { submitAction(); }
        };

    /** The cancel action. */
    protected SimpleAction cancelAction =
        new SimpleAction("Cancel") {
            public void perform() { cancelAction(); }
        };
    
    /*-------------------------------------------------------- 
     Constructor
     *------------------------------------------------------*/
    InGuiRange() { 
        
        /* make a GUI for user input */
        makeDialog();
        
        /* currently not closed */
        closed = false;   
        
        /* extract values for the first balloon */
        next();
    }
        
    /*--------------------------------------------------------
     Methods to implement the IRange interface
     *------------------------------------------------------*/

    /* nullify current data and wait for next submission */
    public synchronized void next() {
    
        submitted = false;
        
        /* wait till user types in the input or closes the GUI */
        while ((!submitted)&&(!closed)) JPTUtilities.pauseThread(10);
    }

    /* return the most recent submission - if available */
    public synchronized Object current() {
           
        if ((submitted) && (!closed)){
            return b;
        }
        else{
            return null;
        }
    }
    
    /* verify that new submission is available */
    public synchronized boolean hasMore() {
        return ((submitted) && (!closed)); 
    }
    

    /*--------------------------------------------------------------
     Methods and helpers to read and close the input
     *------------------------------------------------------------*/
    
    /* save the balloon value for this submission */
    protected void submitAction(){
        b = bView.getBalloon();
        submitted = true;
    }
    
    /* register the closing action */
    protected void cancelAction(){
        closed = true;
        input.dispose();
   }
   
    /*--------------------------------------------------------------
     Make a GUI for user input 
     *------------------------------------------------------------*/
    protected void makeDialog(){
   
           bView = new BalloonView();    
        
        /* create non-modal dialog for balloon inputs */
        input = 
            new GeneralDialog
                 (bView.bView,
                  "Balloon Input",
                  new Object[][]{{"Submit", submitAction, DialogAction.KEEP_OPEN},
                             {"Cancel", cancelAction, DialogAction.AUTO_CLOSE}},
                  false);
                  
        /* specify the closing action */
        input.setWindowClosingAction(cancelAction, DialogAction.AUTO_CLOSE);
        input.setVisible(true); 
    }

}
        
