/*
 * ServerChannel.java @ May 12, 2002
 * by Jason Ansel <jansel@ccs.neu.edu>
 * adapted by Viera K. Proulx @ May 25, 2003
 *
 * This is a Socket wrapper for dealing with sockets that transfer
 * ascii text line by line. It allows for simple collections by both 
 * connecting to a remote server or listening for a connection from
 * a remote computer.  It also provides iterator interfaces for reading
 * text line by line.
 *
 */
 /*
 Example usage:
 
 	// give the desired port to the constructor
 	// optionally, you may also give the host name
	AsciiSocket as=new AsciiSocket(int port);
	AsciiSocket as=new AsciiSocket(String host, int port);

	// open the socket, connect to a server
	// return false if connection fails
	boolean (as.open()) // opens the socket and connects to the port

	// write one line to the socket - no need to put '\n' at the end
	as.write("any--string");
	
	// read one line from the socket
	String inputString = as.read();
	
	// close the socket
	as.close();
	
	// check is server is open
	as.isOpen();

}
 
 
 */
 
import java.net.Socket;
import java.net.ServerSocket;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;

import edu.neu.ccs.console.*;

class ServerChannel implements IOChannel, ConsoleAware{

	/*-------------------------------------------------------------------
	 *  Member Data
	 *-----------------------------------------------------------------*/
	/* the socket adapter for connection from the remote host */
	Socket socket;
		
	/* identification of this host or server */
	String host;
	
	/* the port number used */
	int port = 1234;
	
	/* this is used to prevent multiple threads
				from reading at the same time and breaking lines up*/
	boolean IN_USE=false; 
	
	/* default port used for connecting to remote hosts, 
	   and listening for incoming connections in ServerChannel */
	public static int DEFAULT_PORT=1234; 
	
	/* possibly change to \r\n if you love microsoft */
	public static String NEW_LINE="\n"; 
	
	InputStream in=null;
	OutputStream out=null;
	
	/*-------------------------------------------------------------------
	 *  Constructors
	 *-----------------------------------------------------------------*/
	
	/* constructor for the server connection */
	public ServerChannel(String _host, int _port){
	
		/* just remember the data - wait for the open call */
		this.host = _host;
		this.port = _port;		
	}
	
	/* constructor for the server connection */
	public ServerChannel(int _port){
	
		/* connect to localhost, if no host is given */
		this("localhost", _port);		
	}
	
	/* constructor for the server connection */
	public ServerChannel(){
	
		/* connect to localhost, default port 
		   if no host or port is given */
		this("localhost", DEFAULT_PORT);		
	}
	
	/*-------------------------------------------------------------------
	 *  Internal class
	 *-----------------------------------------------------------------*/
	//thrown if you try to use IO before connectin or after disconnect
	class NotConnectedException extends Exception { }


	/*-------------------------------------------------------------------
	 *  This group of methods is used for opening a server port
	 *-----------------------------------------------------------------*/
	
	/* Listen for exactly 1 incoming connection 
	   on a on given port 
	   Note: will wait untill a connections comes, 
	   can take long time */
	public boolean listenOn(int port){
		try{
			ServerSocket ss=new ServerSocket(port);
			boolean ret=this.listenOn(ss);
			ss.close();
			return ret;
		}
		catch(Exception e){
			return false;
		}
	}

	/* Listen for exactly 1 incoming connection 
	   on a on the default port 
	   Note: will wait untill a connections comes, 
	   can take long time */
	public boolean listenOn(){
		return this.listenOn(DEFAULT_PORT);	
	}
	
	/* Listen for the next incoming connection 
	   on the given server socket.
	   Use this if you expect multiple incoming connections 
	   with multiple Ascii sockets
	   Note: will wait untill a connections comes, 
	   can take long time */
	public boolean listenOn(ServerSocket s){
		try{
			this.socket=s.accept();
			return true;
		}catch(Exception e){
			return false;
		}
	}
	
	/*-------------------------------------------------------------------
	 *  This method is used for opening the socket
	 *-----------------------------------------------------------------*/
		/* open the socket as a server */
	public boolean openServer(){
	
		boolean connected = false;
		
		console.out.println("Server channel listening on port " + port);
		
		/* listen for connection, send acknowledgement */
		if (listenOn(port)){
			writeLine(host + "ready to talk.");
			connected = true;
		}	
		return connected;
	}

	/*-------------------------------------------------------------------
	 *  Populate the in and out streams
	 *-----------------------------------------------------------------*/
	/* ensure that this.in and this.out are correctly populated */
	private void fillStreams() throws NotConnectedException{
	
		if (isOpen()){
			if(in==null||out==null){
				try{
					in= socket.getInputStream();
					out=socket.getOutputStream();
				}catch(Exception e){
					throw(new NotConnectedException());
				}
			}
		}else{
			throw(new NotConnectedException());
		}
	}
	
	/*-------------------------------------------------------------------
	 *  Read from the in stream
	 *-----------------------------------------------------------------*/
	
	/* read 1 ascii char from the socket */
	/* Note: it blocks untill it has something to return */
	public char readChar() throws NotConnectedException{
		int read;
		fillStreams();
		
		try{ 
			read=in.read(); 
		}catch(Exception e){ 
			throw(new NotConnectedException()); 
		}
		
		if(read==-1) 
			throw(new NotConnectedException());
			
		return (char) read;
	}
	
	/* read from the socket untill we reach a given String */
	/* Note: It blocks untill it has something to return */
	public String readTo(String to) throws NotConnectedException{

		try{ while(this.IN_USE) this.wait(100); 
		}catch(Exception e){}
		
		this.IN_USE=true;
		String current="";
		
		while(!current.endsWith(to)){
			current+=this.readChar();
		}
		this.IN_USE=false;
		return current;
	}
	
	/* read the next line from the socket and returned it without the trailing \n */
    /* Note: it blocks untill it has something to return */
    public String readLine() throws NotConnectedException{
    	
    	String line=readTo(NEW_LINE);
    	line=line.substring(0,line.lastIndexOf(NEW_LINE));
    	return line;
    }
    
	/*-------------------------------------------------------------------
	 *  Write to the out stream
	 *-----------------------------------------------------------------*/
	 /* Write a string to this socket */
	 public boolean writeString(String s){
	 	 try{
	 	 	fillStreams();
	 	 	out.write(s.getBytes());
	 	 	return true;
	 	 }catch(Exception e){
	 	 	return false;
	 	 }
	 }
	 
	 /* Write a string to this socket and append NEW_LINE to it */
	 public boolean writeLine(String s){
	 	return writeString(s+NEW_LINE);
	 }
	 
	/*-------------------------------------------------------------------
	 *  Implementing IOChannel interface
	 *-----------------------------------------------------------------*/
    /* open the channel - return true if success */
    public boolean open(){
    
		return openServer();
	}
	
    /* read one line from the channel */
    public String read(){
    	try{
    		return readLine();
    	}
    	catch(Exception e){
    		close();
    	}
    	return "";
    } 
    
    /* write one line to the channel */
    public void write(String line){
    	
    	try{
    		if (!(writeLine(line))){
    			close();
    		}
    	}
    	catch(Exception e){
    		close();
    	}
    } 
        
    /* close the channel */
	public void close(){
		try{
			socket.close();
		}
		catch(Exception e){};
	}
	
    /* determine whether the channel is open */
    public boolean isOpen(){
		return socket.isBound() && 
			   socket.isConnected() && 
			   !socket.isClosed();
    } 
}





