/*
 * SocketChannel.java @ May 12, 2002
 * by Jason Ansel <jansel@ccs.neu.edu>
 * adapted by Viera K. Proulx @ May 25, 2003
 *
 * This is a Socket wrapper for dealing with sockets that transfer
 * ascii text line by line. It allows for simple collections by both 
 * connecting to a remote server or listening for a connection from
 * a remote computer.  It also provides iterator interfaces for reading
 * text line by line.
 *
 */
 /*
 Example usage:
 
 
This code would connect to yahoo.com, make a bad request.. and iterate through yahoo's response line by line
 
AsciiSocket as=new AsciiSocket();
if(as.connectTo("yahoo.com",80)){ //try to connect to yahoo.com
	//now we are connected
	as.writeLine("GET malformed.request HTTP/1.0");
	as.writeLine("");
	for(IRange i=as.lineIRange(); i.hasMore(); i.next()){
		//the current line would be:     i.current()
	}
	//now we are disconnected
}else{
	//could not connect
}
 
 
 
 This code would be for a simple Echo server that repeats all text sent to it:
 
AsciiSocket as=new AsciiSocket();
if(as.listenOn(1234)){ //Try to bind port 1234 and recieve 1 connection from it
	//We are now connected
	//Iterate through incoming lines
	for(IRange i=as.lineIRange(); i.hasMore(); i.next()){
		as.writeLine("You said: "+i.current()); // bounce them back to sender
	}
	//Now disconnected
}else{
	//Couldn't bind socket
}
 
 
 
 
 */
 
import java.net.Socket;
import java.net.ServerSocket;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;

class SocketChannel implements IOChannel{

	/*-------------------------------------------------------------------
	 *  Member Data
	 *-----------------------------------------------------------------*/
	/* the socket adapter for connection to the remote host */
	AsciiSocket as;
	
	/* determine whether we are connected */
	boolean connected = false;
	
	/* identification of this host or client */
	String id;
	
	/* the port number used */
	int port = 1234;
	
	/* true if this is the server */
	boolean isServer = false;
	
	/*-------------------------------------------------------------------
	 *  Constructors
	 *-----------------------------------------------------------------*/

	/* constructor for the server connection */
	public SocketChannel(String _id){
		this(_id, AsciiSocket.DEFAULT_PORT);
		isServer = true;	
	}
	
	/* constructor for the client connection */
	public SocketChannel(String _id, int _port){
	
		/* just remember the data - wait for the open call */
		this.id = _id;
		this.port = _port;
		
	}
	
	/*-------------------------------------------------------------------
	 *  Implementing IOChannel interface
	 *-----------------------------------------------------------------*/
    /* open the channel - return true if success */
    public boolean open(){
    
		if (isServer)
			return openServer();
		else 
			return openClient();
	}
	
	/* open the socket as a client */
	public boolean openClient(){
	
		/* create a new socket */
		as = new AsciiSocket();
				
		/* listen for connection, send acknowledgement */
		if (as.listenOn(port)){
			as.writeLine(id + "ready to talk.");
			connected = true;
		}	
		return connected;
	}

	/* open the socket as a server */
	public boolean openServer(){
		/* create a new socket */
		as = new AsciiSocket();
				
		/* await connection, send acknowledgement */
		if (as.connectTo(id, port)){
			as.writeLine(id + " connected to you.");
			connected = true;
		}
		return connected;
	}    
	
    /* read one line from the channel */
    public String read(){
    	try{
    		return as.readLine();
    	}
    	catch(Exception e){
    		connected = false;
    		close();
    	}
    	return "";
    } 
    
    /* write one line to the channel */
    public void write(String line){
    	
    	try{
    		if (!(as.writeLine(line))){
    			connected = false;
    			close();
    		}
    	}
    	catch(Exception e){
    		connected = false;
    		close();
    	}
    } 
        
    /* close the channel */
	public void close(){
		try{
			as.socket.close();
		}
		catch(Exception e){};
	}
	
    /* determine whether the channel is open */
    public boolean isOpen(){
    	return connected;
    } 

	 
}





