/*
 * AsciiSocket.java @ May 12, 2002
 * by Jason Ansel <jansel@ccs.neu.edu>
 *
 * This is a Socket wrapper for dealing with sockets that transfer
 * ascii text line by line. It allows for simple collections by both 
 * connecting to a remote server or listening for a connection from
 * a remote computer.  It also provides iterator interfaces for reading
 * text line by line.
 *
 */
 /*
 Example usage:
 
 
This code would connect to yahoo.com, make a bad request.. and iterate through yahoo's response line by line
 
AsciiSocket as=new AsciiSocket();
if(as.connectTo("yahoo.com",80)){ //try to connect to yahoo.com
	//now we are connected
	as.writeLine("GET malformed.request HTTP/1.0");
	as.writeLine("");
	for(IRange i=as.lineIRange(); i.hasMore(); i.next()){
		//the current line would be:     i.current()
	}
	//now we are disconnected
}else{
	//could not connect
}
 
 
 
 This code would be for a simple Echo server that repeats all text sent to it:
 
AsciiSocket as=new AsciiSocket();
if(as.listenOn(1234)){ //Try to bind port 1234 and recieve 1 connection from it
	//We are now connected
	//Iterate through incoming lines
	for(IRange i=as.lineIRange(); i.hasMore(); i.next()){
		as.writeLine("You said: "+i.current()); // bounce them back to sender
	}
	//Now disconnected
}else{
	//Couldnt bind socket
}
 
 
 
 
 */
 
import java.net.Socket;
import java.net.ServerSocket;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;

class AsciiSocket{

	/*******************************************************************
	 *  Member Data
	 *******************************************************************/
	Socket socket; //Connection to remote host
	
	boolean IN_USE=false; /* this is used to prevent multiple threads
				from reading at the same time and breaking lines up*/
	
	public static int DEFAULT_PORT=1234; /* default port used for both
				connecting to remote hosts, and listening for incoming
				connections */
	
	public static String NEW_LINE="\n"; /* possible change to \r\n if 
				you love microsoft */
	
	InputStream in=null;
	OutputStream out=null;
	
	
	/*******************************************************************
	 *  Constructors
	 *******************************************************************/		

	/* wrap around a given socket */
	AsciiSocket(Socket s){
		this.socket=s;
	}
	
    /* unconnected socket */
	AsciiSocket(){
		this(new Socket());
	}
	
	/*******************************************************************
	 *  Internal classes
	 *******************************************************************/
	//thrown if you try to use IO before connectin or after disconnect
	class NotConnectedException extends Exception { }
	//Iterator for traversing lines coming from a asciisocket
	class AsciiSocketIterator implements Iterator{
		AsciiSocket as;
		Object nextItem=null;
		boolean hasNext;
		AsciiSocketIterator(AsciiSocket as){
			this.as=as;
		}
		
		/* check if there is more to read */
		public boolean hasNext(){
			if(nextItem==null){
				try{
					nextItem=as.readLine();
					hasNext=true;
				}catch(NotConnectedException e){
					nextItem=null;
					hasNext=false;
				}
			}
			return hasNext;

		}
		
		/* get next item */
		public Object next(){
			Object ret=nextItem;
			nextItem=null;
			return ret;
		}
		
		/* remove not supported */
		public void remove() throws UnsupportedOperationException{ throw(new UnsupportedOperationException()); }
		
	}
	
	
	/*******************************************************************
	 *  This group of methods is used for connecting to a remote host
	 *  both via listening for a connection and making one
	 *******************************************************************/
	
	/* make this a connection to a given remote host on a given port */
	public boolean connectTo(String host,int port){
		try{
			this.socket=new Socket(host,port);
			return true;
		}catch(Exception e){
			return false;
		}
	}
	
	
	/* make this a connection to a given remote host on the default port */
	public boolean connectTo(String host){
		return this.connectTo(host,DEFAULT_PORT);	
	}
	
	
	/* Listen for exactly 1 incoming connection on a on given port */
	//Note: will wait untill a connections comes, can take long time
	public boolean listenOn(int port){
		try{
			ServerSocket ss=new ServerSocket(port);
			boolean ret=this.listenOn(ss);
			ss.close();
			return ret;
		}catch(Exception e){
			return false;
		}
	}

	/* Listen for exactly 1 incoming connection on the default port */
	//Note: will wait untill a connections comes, can take long time
	public boolean listenOn(){
		return this.listenOn(DEFAULT_PORT);	
	}
	
	/* Listen for the next incoming connection on the given server socket*/
	//Use this if you expect multiple incoming connections with multiple Ascii sockets
	//Note: will wait untill a connections comes, can take long time
	public boolean listenOn(ServerSocket s){
		try{
			this.socket=s.accept();
			return true;
		}catch(Exception e){
			return false;
		}
	}
	
	/* close the connection */
	public void close(){
		try{
			this.socket.close();
		}
		catch(Exception e){
			System.out.println(e.getMessage());
		}
	}
	
	/*******************************************************************
	 *  This group of methods is for reading
	 *******************************************************************/
	
	/* ensure that this.in and this.out are correctly populated */
	private void fillStreams() throws NotConnectedException{
		if(socket.isBound() && socket.isConnected() && !socket.isClosed()){
			if(in==null||out==null){
				try{
					in= socket.getInputStream();
					out=socket.getOutputStream();
				}catch(Exception e){
					throw(new NotConnectedException());
				}
			}
		}else{
			throw(new NotConnectedException());
		}
	}
	
	/* read 1 ascii char from the socket */
	//Note blocks untill it has something to return
	public char readChar() throws NotConnectedException{
		int read;
		fillStreams();
		try{ read=in.read(); }
		catch(Exception e){ throw(new NotConnectedException()); }
		if(read==-1) throw(new NotConnectedException());
		return (char) read;
	}
	
	/* read from the socket untill we reach a given String */
	//Note blocks untill it has something to return
	public String readTo(String to) throws NotConnectedException{
		try{ while(this.IN_USE) this.wait(100); }catch(Exception e){}
		this.IN_USE=true;
		String current="";
		while(!current.endsWith(to)){
			current+=this.readChar();
		}
		this.IN_USE=false;
		return current;
	}
	
	/* read the next line from the socket and returned it without the trailing \n */
    //Note blocks untill it has something to return
    public String readLine() throws NotConnectedException{
    	String line=readTo(NEW_LINE);
    	line=line.substring(0,line.lastIndexOf(NEW_LINE));
    	return line;
    }
    
    /*******************************************************************
	 *  This group of methods is for writing
	 *******************************************************************/
	 
	 /* Write a string to this socket */
	 public boolean write(String s){
	 	 try{
	 	 	fillStreams();
	 	 	out.write(s.getBytes());
	 	 	return true;
	 	 }catch(Exception e){
	 	 	return false;
	 	 }
	 }
	 
	 /* Write a string to this socket and append NEW_LINE to it */
	 public boolean writeLine(String s){
	 	return write(s+NEW_LINE);
	 }
	 
    /*******************************************************************
	 *  Iteratators for reading from this socket line by line
	 *******************************************************************/
	 
	 /* Return a java iterator over incoming lines in this socket */
	 Iterator lineIterator(){
		 return new AsciiSocketIterator(this);
	 }
	 
	 /* return a com1201 style IRange for reading lines*/
	 IRange lineIRange(){
		 return new IteratorIRange(this.lineIterator());
	 }
	 
}





