/*
 * @(#)MethodGUIalt.java  1.0  4 September 2002
 *
 * Copyright 2001,2002
 * College of Computer Science
 * NORTH_EASTeastern University
 * Boston, MA  02115
 *
 * This software may be used for educational purposes as long as
 * this copyright notice is retained intact at the top of all files.
 *
 * Should this software be modified, the words "Modified from 
 * Original" must be included as a comment below this notice.
 *
 * All publication rights are retained.  This software or its 
 * documentation may not be published in any media either in whole
 * or in part without explicit permission.
 *
 * Contact information:
 *   Richard Rasala    rasala@ccs.neu.edu
 *   Viera Proulx      vkp@ccs.neu.edu
 *   Jeff Raab         jmr@ccs.neu.edu
 * 
 * Telephone:          617-373-2462
 *
 * This software was created with support from NORTH_EASTeastern 
 * University and from NSF grant DUE-9950829.
 */

import edu.neu.ccs.*;
import edu.neu.ccs.gui.*;
import edu.neu.ccs.codec.*;
import edu.neu.ccs.console.*;
import edu.neu.ccs.filter.*;
import edu.neu.ccs.parser.*;
import edu.neu.ccs.pedagogy.*;
import edu.neu.ccs.util.*;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.util.*;
import java.text.*;
import java.math.*;
import javax.swing.*;
import java.lang.reflect.*;

/**
 * This class implements the GUI for a Method that is non-trivial,
 * that is, has a return value and/or non-trivial parameters.
 *
 * @author Viera Proulx
 * @author Richard Rasala
 */
public class MethodGUIalt extends TablePanel
    implements JPTConstants, ConsoleAware
{   
    /** The associated JPFApplicationAlt.*/
    JPFApplicationAlt application = null;
    
    /** The method used to define this MethodGUI. */
    protected Method method = null;
    
    // Method Information
    
    /** The method name. */
    protected String name = null;
    
    /** The return type. */
    protected Class returnType = null;
    
    /** The return type name. */
    protected String returnTypeName = null;
    
    /** The parameter types. */
    protected Class[] parameterTypes = null;
    
    /** The parameter type names. */
    protected String[] parameterTypeNames = null;
    
    // GUI Information
    
    /** The MethodGUI orientation. */
    protected int orientation = HORIZONTAL;
    
    /** The default text field view size. */
    protected final int TFVSize = 150;
    
    /** The return view. */
    TypedView returnView = null;
    
    /** The parameter views. */
    TypedView[] parameterViews = null;
    
    // The evaluate action
    Action evaluate = null;
    
    //////////////////
    // Constructors //
    //////////////////
    
    /** Constructs the MethodGUI using the given Method. */
    protected MethodGUIalt(JPFApplicationAlt application, Method method) {
        this(application, method, HORIZONTAL);
    }
    
    /** Constructs the MethodGUIalt using the given Method and orientation. */
    protected MethodGUIalt(JPFApplicationAlt application, Method method, int orientation) {
        super(0, 0, 10, 10, CENTER);
        
        if (method == null)
            return;
        
        this.application = application;
        this.method = method;
        this.orientation = orientation;
        
        extractInformation();
        createEvaluate();
        buildTable();
    }
    
    ///////////////////////
    // Protected Methods //
    ///////////////////////
    
    /** Extract the important information about the method. */
    protected void extractInformation() {
        name = method.getName();
        
        returnType = method.getReturnType();
        
        returnTypeName = application.className(returnType);
        
        parameterTypes = method.getParameterTypes();
        
        parameterTypeNames = new String[parameterTypes.length];
        
        int length = parameterTypes.length;
        
        for (int i = 0; i < length; i++){
            parameterTypeNames[i] = application.className(parameterTypes[i]);
        }
    }
    
    /** Build the GUI TablePanel. */
    protected void buildTable() {
        // make sure that orientation has a valid value
        if (orientation != VERTICAL)
            orientation = HORIZONTAL;
        
        // define rows and cols for default orientation of HORIZONTAL
        int rows = 3;
        int cols = parameterTypes.length
            + 1
            + (returnType.equals(void.class) ? 0 : 1);
        
        // if there are too many cols, make the orientation VERTICAL
        if (cols > 6)
            orientation = VERTICAL;
        
        // if orientation is VERTICAL then interchange rows and cols
        if (orientation == VERTICAL) {
            int temp = cols;
            cols = rows;
            rows = temp;
        }
        
        // now set the rows and cols of the table
        setRows(rows);
        setColumns(cols);
        
        buildParameterViews();
        buildReturnView();
        
        if (orientation == VERTICAL)
            buildVerticalGUI();
        else
            buildHorizontalGUI();
    }
    
    /** Install the views in the GUI using a vertical arrangement. */
    protected void buildVerticalGUI() {
        int length = parameterTypes.length;
        int row;
        
        for (row = 0; row < length; row++) {
            addObject(parameterTypeNames[row], row, 0);
            addObject(createInputIcon(),       row, 1);
            addObject(parameterViews[row],     row, 2);
        }
        
        row = length;
        
        addObject(name,               row, 0);
        addObject(createActionIcon(), row, 1);
        addObject(evaluate,           row, 2);
        
        if (! returnType.equals(void.class)) {
            row = length + 1;
            
            addObject(returnTypeName,     row, 0);
            addObject(createOutputIcon(), row, 1);
            addObject(returnView,         row, 2);
        }
    }
    
    /** Install the views in the GUI using a horizontal arrangement. */
    protected void buildHorizontalGUI() {
        int length = parameterTypes.length;
        int next = 0;
        int col;
        
        if (! returnType.equals(void.class)) {
            addObject(returnTypeName,     0, 0);
            addObject(createOutputIcon(), 1, 0);
            addObject(returnView,         2, 0);
            
            next = 1;
        }
        
        col = next;
        
        addObject(name,               0, col);
        addObject(createActionIcon(), 1, col);
        addObject(evaluate,           2, col);
        
        next++;
        
        for (col = 0; col < length; col++) {
            addObject(parameterTypeNames[col], 0, col + next);
            addObject(createInputIcon(),       1, col + next);
            addObject(parameterViews[col],     2, col + next);
        }
    }
    
    /** Return a typed view appropriate for the given class. */
    protected TypedView createView(Class c) {
        if (c == null)
            return null;
        
        if (c.equals(Color.class) || c.equals(XColor.class)) {
            return new ColorView(Color.black, true);
        }
        
        TextFieldView tfv
            = new TextFieldView("", getErrorPrompt(c), "InputError", TFVSize);
        
        if (Stringable.class.isAssignableFrom(c))
            tfv.setDataType(c);
        
        else if (c.isPrimitive()) {
            if (c.equals(byte.class))
                tfv.setDataType(XByte.class);
            else if (c.equals(short.class))
                tfv.setDataType(XShort.class);
            else if (c.equals(int.class))
                tfv.setDataType(XInt.class);
            else if (c.equals(long.class))
                tfv.setDataType(XLong.class);
            else if (c.equals(float.class))
                tfv.setDataType(XFloat.class);
            else if (c.equals(double.class))
                tfv.setDataType(XDouble.class);
            else if (c.equals(char.class))
                tfv.setDataType(XChar.class);
            else if (c.equals(boolean.class))
                tfv.setDataType(XBoolean.class);
        }
        
        else if (c.equals(String.class))
            tfv.setDataType(XString.class);
        
        else if (c.equals(BigInteger.class))
            tfv.setDataType(XBigInteger.class);
        
        else if (c.equals(BigDecimal.class))
            tfv.setDataType(XBigDecimal.class);
        
        else if (c.equals(Point2D.Double.class))
            tfv.setDataType(XPoint2D.class);
        
        return tfv;
    }
    
    /** Build the array of views for the method parameters. */
    protected void buildParameterViews() {
        int length = parameterTypes.length;
        
        parameterViews = new TypedView[length];
        
        for (int i = 0; i < length; i++)
            parameterViews[i] = createView(parameterTypes[i]);
    }
    
    /** Build the view for the return value. */
    protected void buildReturnView() {
        returnView = createView(returnType);
    }
    
    /** Return an error prompt customized for the given class. */
    protected String getErrorPrompt(Class c) {
        return "Error in data of type " + application.className(c);
    }
    
    protected JComponent createInputIcon() {
        return new ActivityIconAlt.InputIcon();
    }
    
    protected JComponent createActionIcon() {
        return new ActivityIconAlt.ActionIcon();
    }
    
    protected JComponent createOutputIcon() {
        return new ActivityIconAlt.OutputIcon();
    }
    
    /**
     * Return the user input as an object
     * from the typed view being used to
     * obtain data for the given class.
     */
    protected Object extractParameterValue(TypedView view, Class c) {
        if (view == null)
            return null;
        
        Object object = view.demandObject();
        
        if (object instanceof XByte) {
            if (c.equals(byte.class))
                object = new Byte       (((XByte)object)   .getValue());
        }
        
        else
        if (object instanceof XShort){
            if (c.equals(short.class))
                object = new Short      (((XShort)object)  .getValue());
        }
        
        else
        if (object instanceof XInt) {
            if (c.equals(int.class))
                object = new Integer    (((XInt)object)        .getValue());
        }
        
        else
        if (object instanceof XLong) {
            if (c.equals(long.class))
                object = new Long       (((XLong)object)       .getValue());
        }
        
        else
        if (object instanceof XFloat) {
            if (c.equals(float.class))
                object = new Float      (((XFloat)object)      .getValue());
        }
        
        else
        if (object instanceof XDouble) {
            if (c.equals(double.class))
                object = new Double     (((XDouble)object)     .getValue());
        }
        
        else
        if (object instanceof XChar) {
            if (c.equals(char.class))
                object = new Character  (((XChar)object)       .getValue());
        }
        
        else
        if (object instanceof XBoolean) {
            if (c.equals(boolean.class))
                object = new Boolean    (((XBoolean)object)    .getValue());
        }
        
        else
        if (object instanceof XString) {
            if (c.equals(String.class))
                object = new String     (((XString)object)     .getValue());
        }
        
        else
        if (object instanceof XColor) {
            if (c.equals(Color.class))
                object = ((XColor)object).getValue();
        }
        
        else
        if (object instanceof XBigInteger) {
            if (c.equals(BigInteger.class))
                object = ((XBigInteger)object) .getValue();
        }
        
        else
        if (object instanceof XBigDecimal) {
            if (c.equals(BigDecimal.class))
                object = ((XBigDecimal)object) .getValue();
        }
        
        return object;
    }
    
    /** Return the array of user input values from the parameter views. */
    protected Object[] extractParameterValues() {
        int length = parameterViews.length;
        
        Object[] values = new Object[length];
        
        for (int i = 0; i < length; i++)
            values[i] = extractParameterValue(parameterViews[i], parameterTypes[i]);
        
        return values;
    }
    
    /**
     * Display the return value in the return view.
     * For most types, this will display the result of the toString() method.
     */
    protected void showReturnValue(Object value) {
        if (value == null) {
            returnView.setViewState("null");
            return;
        }
        
        // handle colors as a special case
        if (value instanceof XColor)
            value = ((XColor) value).getValue();
        
        if (value instanceof Color) {
            Color c = (Color) value;
            ColorView v = (ColorView) returnView;
            
            v.setColor(c);
            return;
        }
        
        // convert Point2D.Double to XPoint2D to obtain nicer toString method
        if (value instanceof Point2D.Double) {
            Point2D.Double p = (Point2D.Double) value;
            value = new XPoint2D(p.getX(), p.getY());
        }
        
        returnView.setViewState(value.toString());
    }
    
    /** Create the evaluate action for the Evaluate button in the GUI. */
    protected void createEvaluate() {
        evaluate = new SimpleAction("Evaluate") {
            public void perform() {
                synchronized(application) {
                    evaluate();
                }
            }
        };
    }
    
    /** The evaluate method executed by the evaluate action. */
    protected void evaluate() {
        if (returnType.equals(void.class))
            evaluateVoid();
        else
            evaluateWithReturn();
    }
    
    /** The evaluate method in the case of void return. */
    protected void evaluateVoid() {
        try {
            if (application.isStatic(method))
                method.invoke(null, extractParameterValues());
            else
                method.invoke(application.initializer, extractParameterValues());
        }
        catch (Exception exception) {
            application.handleMethodException(exception, name);
        }
    }
    
    /** The evaluate method in the case of non-void return. */
    protected void evaluateWithReturn() {
        Object value = null;
        
        try {
            if (application.isStatic(method))
                value
                    = method.invoke(null, extractParameterValues());
            else
                value
                    = method.invoke(application.initializer, extractParameterValues());
            
            showReturnValue(value);
        }
        catch (Exception exception) {
            application.handleMethodException(exception, name);
        }
    }
    
}