/*
 * @(#)JPFApplicationAlt.java  1.0  4 September 2002
 *
 * Copyright 2001,2002
 * College of Computer Science
 * Northeastern University
 * Boston, MA  02115
 *
 * This software may be used for educational purposes as long as
 * this copyright notice is retained intact at the top of all files.
 *
 * Should this software be modified, the words "Modified from 
 * Original" must be included as a comment below this notice.
 *
 * All publication rights are retained.  This software or its 
 * documentation may not be published in any media either in whole
 * or in part without explicit permission.
 *
 * Contact information:
 *   Richard Rasala    rasala@ccs.neu.edu
 *   Viera Proulx      vkp@ccs.neu.edu
 *   Jeff Raab         jmr@ccs.neu.edu
 * 
 * Telephone:          617-373-2462
 *
 * This software was created with support from Northeastern 
 * University and from NSF grant DUE-9950829.
 */

import edu.neu.ccs.*;
import edu.neu.ccs.gui.*;
import edu.neu.ccs.codec.*;
import edu.neu.ccs.console.*;
import edu.neu.ccs.filter.*;
import edu.neu.ccs.parser.*;
import edu.neu.ccs.pedagogy.*;
import edu.neu.ccs.util.*;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.font.*;
import java.util.*;
import java.text.*;
import java.math.*;
import javax.swing.*;
import java.lang.reflect.*;

/**
 * <P>This is the main application that sets up a console, a graphics window,
 * and installs a list of actions to select one of several tasks to run from
 * a given collection of "proper" methods.</P>
 *
 * @author Viera Proulx
 * @author Richard Rasala
 * @author Jason Jay Rodrigues
 * @author Jeff Raab
 */
public class JPFApplicationAlt extends DisplayPanel
    implements JPTConstants, ConsoleAware
{   
    /** The default buffer width. */
    public static final int BUFFER_WIDTH  = 400;
    
    /** The default buffer height. */
    public static final int BUFFER_HEIGHT = 400;
    
    /** The default graphics buffered panel. */
    protected BufferedPanel window = new BufferedPanel(BUFFER_WIDTH, BUFFER_HEIGHT);
    
    /** The initializer object for the class with the methods. */
    protected JPFalt initializer = null;
    
    /** The class of the initializer object. */
    protected Class initializerClass = null;
    
    /** The JPT frame. */
    protected JPTFrame frame = null;
    
    /** The window frame title. */
    protected String frameTitle = null;
               
    /** If true, show graphics window. */
    protected boolean showGraphicsWindow = true;
    
    /** The maximum number of rows in the buttons panel. */
    protected final int MAXIMUM_ROWS = 25;
    
    /** The buttons panel. */
    protected ActionsPanel buttons
        = new ActionsPanel(
            new TableLayout(MAXIMUM_ROWS, 1, 0, 0, CENTER, HORIZONTAL)
        );
    
    /** The LHS of main GUI panel. */
    protected DisplayPanel LHS = null;
    
    /** The RHS of main GUI panel. */
    protected DisplayPanel RHS = null;
    
    /** The main GUI panel. */
    protected TablePanel mainPanel = null;
    
    /////////////////
    // Constructor //
    /////////////////
    
    /**
     * The protected constructor that constructs the GUI.  This
     * class must be constructed using <CODE>JPFBase</CODE>.
     */
    protected JPFApplicationAlt(JPFalt initializer, String title) {
        if (initializer == null)
            return;
        
        this.initializer = initializer;
        this.initializerClass = initializer.getClass();
        
        setFrameTitle(title);
        
        buildButtons();
        buildGUI();
        
        frame = JPTFrame.createQuickJPTFrame(
            getFrameTitle(),
            this,
            new CenterLayout(),
            NORTH_EAST);
    }
    
    ////////////////
    // Public API //
    ////////////////
    
    /** Return the graphics window. */
    public BufferedPanel getGraphicsWindow() {
        return window;
    }
    
    /** Return the title for the framework window. */
    public String getFrameTitle() {
        return frameTitle;
    }
    
    /** Set the title for the framework window. */
    public void setFrameTitle(String title) {
        frameTitle = (title != null) ? title : className(initializerClass);
        
        if (frame != null)
            frame.setTitle(frameTitle);
    }
    
    ////////////////////////////////
    // Protected Member Functions //
    ////////////////////////////////
    
    /** Build the buttons. */
    protected void buildButtons() {
        Method[] methods = getMethodList();
        int length = methods.length;
        
        for (int i = 0; i < length; i++) {
            if (methods[i] == null)
                continue;
            
            addOneButton(
                makeActionFromMethod(methods[i]),
                getMethodToolTip(methods[i]));
        }
        
        // add the Clear Graphics button
        addOneButton(
            new SimpleAction("Clear Graphics") 
                { public void perform () { clearGraphics(); } },
            "Clear the Graphics Window");

        // add the Toggle Graphics button
        addOneButton(
            new SimpleAction("Toggle Graphics")
                { public void perform () { toggleGraphics(); } },
            "Show or Hide the Graphics Window");

        // add the Exit Framework button
        addOneButton(
            new SimpleAction("Exit") 
                { public void perform () { exitFramework(); } },
            "Exit the Framework");
        
        // uniformize the button sizes
        buttons.uniformizeSize();
    }
    
    /** The clearGraphics method for the Clear Graphics action. */
    protected void clearGraphics() {
        window.clearPanel();
        window.repaint();
    }
    
    /** Toggle the visibility of the graphics window. */
    protected void toggleGraphics() {
        if (showGraphicsWindow)
            mainPanel.remove(RHS);
        else
            mainPanel.add(RHS);
        
        showGraphicsWindow = !showGraphicsWindow;
        
        frame.pack();
        frame.setLocation(NORTH_EAST);
    }
    
    /** The exitFramework method for the Exit action. */
    protected void exitFramework() {
        console.setActivated(false);
        System.exit(0); 
    }
    
    /** Add one button with a tool tip to the buttons panel. */
    protected void addOneButton(Action action, String tooltip) {
        buttons.addAction(action);
        
        buttons.findMatchedButton(action).setToolTipText(tooltip);
    }
    
    /** Return a tool tip appropriate for this method. */
    protected String getMethodToolTip(Method method) {
        if (method == null)
            return "";
        
        return
            ((isStatic(method)) ? "static " : "")
            + className(method.getReturnType())
            + " "
            + method.getName()
            + "("
            + getParameterNames(method)
            + ")"
            + " in "
            + className(method.getDeclaringClass());
    }
    
    /** Return a String representing the parameters of the method. */
    protected String getParameterNames(Method method) {
        String result = "";
        
        if (method == null)
            return result;
        
        Class[] types = method.getParameterTypes();
        
        int length = types.length;
        
        if (length == 0)
            return result;
        
        result = className(types[0]);
        
        for (int i = 1; i < length; i++)
            result += ", " + className(types[i]);
        
        return result;
    }
    
    /** Build the main panel. */
    protected void buildGUI() {
        showConsole();
        
        LHS = new Display(buttons, null, "Tasks");
        
        RHS = new Display(window,  null, "Graphics");
        
        mainPanel = new TablePanel(
            new Object[] { LHS, RHS }, HORIZONTAL, 5, 5, NORTH);
        
        add(mainPanel);
    }
    
    /** Show the console using color text. */
    protected void showConsole() {
        console.setActivated(true);
        console.selectColorTextScheme();
    }
    
    /**
     * Return a list of the "proper" methods, that is, methods for which
     * we can instantiate buttons in the main GUI.  This list may contain
     * <CODE>null</CODE>s.
     */
    protected Method[] getMethodList(){
        Method[] methods =
            getExtraMethods(initializerClass, JPFalt.class);
        
        int length = methods.length;
        
        for (int i = 0; i < length; i++)
            if (! (isSimpleMethod(methods[i]) || isGUIMethod(methods[i])))
                methods[i] = null;
        
        removeDuplicateVirtualMethods(methods);
        
        return methods;
    }
    
    /** Remove duplicate virtual methods defined in several classes. */
    protected void removeDuplicateVirtualMethods(Method[] methods) {
        if (methods == null)
            return;
        
        int length = methods.length;
        
        for (int i = (length - 1); i > 0; i--) {
            if (methods[i] == null)
                continue;
            
            if (isStatic(methods[i]))
                continue;
            
            for (int j = (i - 1); j >= 0; j--) {
                if (methods[j] == null)
                    continue;
                
                if (isStatic(methods[j]))
                    continue;
                
                if (isDuplicate(methods[i], methods[j]))
                    methods[j] = null;
            }
        }
    }
    
    /** Return true if the two methods have the same signature. */
    protected boolean isDuplicate(Method method1, Method method2) {
        if ((method1 == null) || (method2 == null))
            return false;
        
        return
            method1.getName().equals(method2.getName())
            && (method1.getModifiers() == method2.getModifiers())
            && (method2.getReturnType() == method2.getReturnType())
            && isDuplicateParameterList(method1, method2);
    }
    
    /** Return true if the two methods have the same parameter list. */
    protected boolean isDuplicateParameterList(Method method1, Method method2) {
        if ((method1 == null) || (method2 == null))
            return false;
        
        Class[] types1 = method1.getParameterTypes();
        Class[] types2 = method2.getParameterTypes();
        
        int length1 = types1.length;
        int length2 = types2.length;
        
        if (length1 != length2)
            return false;
        
        for (int i = 0; i <length1; i++)
            if (types1[i] != types2[i])
                return false;
        
        return true;
    }
    
    /** Make a ThreadedAction from the given method. */
    protected Action makeActionFromMethod(final Method method) {
        if (method == null)
            return null;
        
        final String name = (isStatic(method))
            ? className(method.getDeclaringClass()) + "." + method.getName()
            : method.getName();
        
        return new ThreadedAction(
            new SimpleAction(name) {
                public void perform () { performAction(method); }
            });    
    }
    
    /** Assure that the method calls are synchronized */
    protected synchronized void performAction(Method method){
        String name = method.getName();
        
        try {
            if (isGUIMethod(method))
                performActionUsingGUI(method);
            else if (isStatic(method))
                method.invoke(null, null);
            else
                method.invoke(initializer, null);
        }
        catch (Exception exception) {
            handleMethodException(exception, name);
        }
    }
    
    /** Handle exception produced by Method invocation. */
    protected void handleMethodException(Exception exception, String name) {
    
        if (exception instanceof IllegalAccessException){
            IllegalAccessException illAccEx
                = (IllegalAccessException) exception;
            
            console.err.print  ("IllegalAccessException: ");
            console.err.println("" + illAccEx); 
            console.err.println("In: " + name + "\n");
        }
        else if (exception instanceof IllegalArgumentException){
            IllegalArgumentException illArgEx
                = (IllegalArgumentException) exception;
            
            console.err.print  ("IllegalArgumentException: ");
            console.err.println("" + illArgEx);
            console.err.println("In: " + name + "\n");
        }
        else if (exception instanceof InvocationTargetException){
            InvocationTargetException invTarEx
                = (InvocationTargetException) exception;
            
            Throwable target = invTarEx.getTargetException();
            
            console.err.println
                ("Exception: " + target + "\nIn: " + name);

            StackTraceElement[] stacktrace = target.getStackTrace();
            
            if (stacktrace != null) {
                int length = stacktrace.length;
                
                for (int i = 0; i< length; i++) {
                    String s = stacktrace[i]  + "";
                    
                    if (s.startsWith("sun.reflect."))
                        break;
                    
                    console.err.println("    at: " + s);
                }
                
                console.err.println();
            }
        }
        else {
            console.err.println
                ("Exception: " + exception + "\nIn: " + name + "\n");
        }
    }
    
    public synchronized void performActionUsingGUI(Method method){
        JPTFrame.createQuickJPTFrame(
            method.getName() + " evaluator",
            new MethodGUIalt(this, method),
            new CenterLayout(),
            CENTER);
    }

    /** Return true if the method should be converted to a simple button. */
    protected boolean isSimpleMethod(Method method) {
        if (method == null)                             // no method passed
            return false;
        
        if (method.getName().equals("main"))            // exclude any main
            return false;
        
        if (method.getReturnType() != void.class)       // method is not void
            return false;
        
        if (method.getParameterTypes().length > 0)      // method has parameters
            return false;
        
        int modifiers = method.getModifiers();
            
        return (((modifiers & Modifier.PROTECTED) == 0)
                && ((modifiers & Modifier.PRIVATE) == 0)); // method is public
        
    }
    
    /** Return true if the method should be converted to a button. with GUI */
    protected boolean isGUIMethod(Method method) {
        if (method == null)                             // no method passed
            return false;
        
        if (method.getName().equals("main"))            // exclude any main
            return false;
        
        if (! isAcceptableMethodForGUI(method))         // exclude methods that
            return false;                               // we cannot handle in GUI
        
        int modifiers = method.getModifiers();
            
        return (((modifiers & Modifier.PROTECTED) == 0)
                && ((modifiers & Modifier.PRIVATE) == 0)); // method is public
    }
    
    /** Return true if the type is acceptable for the automatic MethodGUI. */
    protected boolean isAcceptableTypeForGUI(Class c) {
        if (c == null)
            return false;
        
        if (c.isPrimitive())
            return true;
        
        if (Stringable.class.isAssignableFrom(c))
            return true;
        
        if (c.equals(String.class))
            return true;
        
        if (c.equals(Color.class))
            return true;
        
        if (c.equals(BigInteger.class))
            return true;
        
        if (c.equals(BigDecimal.class))
            return true;
        
        if (c.equals(Point2D.Double.class))
            return true;
        
        return false;
    }
    
    /** 
     * Return true if the method types are acceptable for the automatic MethodGUI
     * and if the method is either non-void or has parameters.
     */
    protected boolean isAcceptableMethodForGUI(Method method) {
        if (method == null)
            return false;
        
        Class returnType = method.getReturnType();
        Class[] parameterTypes = method.getParameterTypes();
        
        int length = parameterTypes.length;
        
        if (returnType.equals(void.class) && (length == 0))
            return false;
        
        for (int i = 0; i < length; i++)
            if (! isAcceptableTypeForGUI(parameterTypes[i]))
                return false;
        
        return true;
    }
    
    /** Get the methods in Class c that do not come from Class d. */
    protected Method[] getExtraMethods(Class c, Class d) {
        if ((c == null)
            || (c.equals(d))
            || (c.isPrimitive())
            || (c.isInterface())
            || (c.isArray()))
                return new Method[] { };
        
        return joinMethodArrays(
            getExtraMethods(c.getSuperclass(), d),
            c.getDeclaredMethods());
    }
    
    /** Join two Method arrays into one. */
    protected Method[] joinMethodArrays(Method[] listA, Method[] listB){
    
        int lengthA = listA.length;
        int lengthB = listB.length;
        int length = lengthA + lengthB;
        
        Method[] list = new Method[length];
        
        for (int i = 0; i < lengthA; i++)
            list[i] = listA[i];
        
        for (int i = lengthA; i < length; i++)
            list[i] = listB[i - lengthA];
        
        return list;
    }
    
    /* Return whether or not the method is static. */
    protected boolean isStatic(Method method) {
        if (method == null)
            return false;
        
        int modifiers = method.getModifiers();
        
        return (modifiers & Modifier.STATIC) != 0;
    }
    
    /** Returns the unqualified name of the given Class */
    protected String className(Class c) {
        String s = c.getName();
        
        int pos = s.lastIndexOf(".");
        
        if (pos < 0)
            return s;
        
        return s.substring(pos + 1);
    }
    
}
