/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.shuffle;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import org.apache.spark.network.buffer.FileSegmentManagedBuffer;
import org.apache.spark.network.buffer.ManagedBuffer;
import org.apache.spark.network.shuffle.protocol.ExecutorShuffleInfo;
import org.apache.spark.network.util.JavaUtils;
import org.apache.spark.network.util.LevelDBProvider;
import org.apache.spark.network.util.NettyUtils;
import org.apache.spark.network.util.TransportConf;
import org.iq80.leveldb.DB;
import org.iq80.leveldb.DBIterator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spark_project.guava.annotations.VisibleForTesting;
import org.spark_project.guava.base.Objects;
import org.spark_project.guava.collect.Maps;

public class ExternalShuffleBlockResolver {
    private static final Logger logger = LoggerFactory.getLogger(ExternalShuffleBlockResolver.class);
    private static final ObjectMapper mapper = new ObjectMapper();
    private static final String APP_KEY_PREFIX = "AppExecShuffleInfo";
    private static final LevelDBProvider.StoreVersion CURRENT_VERSION = new LevelDBProvider.StoreVersion(1, 0);
    @VisibleForTesting
    final ConcurrentMap<AppExecId, ExecutorShuffleInfo> executors;
    private final Executor directoryCleaner;
    private final TransportConf conf;
    @VisibleForTesting
    final File registeredExecutorFile;
    @VisibleForTesting
    final DB db;
    private final List<String> knownManagers = Arrays.asList("org.apache.spark.shuffle.sort.SortShuffleManager", "org.apache.spark.shuffle.unsafe.UnsafeShuffleManager");

    public ExternalShuffleBlockResolver(TransportConf conf, File registeredExecutorFile) throws IOException {
        this(conf, registeredExecutorFile, Executors.newSingleThreadExecutor(NettyUtils.createThreadFactory((String)"spark-shuffle-directory-cleaner")));
    }

    @VisibleForTesting
    ExternalShuffleBlockResolver(TransportConf conf, File registeredExecutorFile, Executor directoryCleaner) throws IOException {
        this.conf = conf;
        this.registeredExecutorFile = registeredExecutorFile;
        this.db = LevelDBProvider.initLevelDB((File)this.registeredExecutorFile, (LevelDBProvider.StoreVersion)CURRENT_VERSION, (ObjectMapper)mapper);
        this.executors = this.db != null ? ExternalShuffleBlockResolver.reloadRegisteredExecutors(this.db) : Maps.newConcurrentMap();
        this.directoryCleaner = directoryCleaner;
    }

    public void registerExecutor(String appId, String execId, ExecutorShuffleInfo executorInfo) {
        AppExecId fullId = new AppExecId(appId, execId);
        logger.info("Registered executor {} with {}", (Object)fullId, (Object)executorInfo);
        if (!this.knownManagers.contains(executorInfo.shuffleManager)) {
            throw new UnsupportedOperationException("Unsupported shuffle manager of executor: " + executorInfo);
        }
        try {
            if (this.db != null) {
                byte[] key = ExternalShuffleBlockResolver.dbAppExecKey(fullId);
                byte[] value = mapper.writeValueAsString((Object)executorInfo).getBytes(StandardCharsets.UTF_8);
                this.db.put(key, value);
            }
        }
        catch (Exception e) {
            logger.error("Error saving registered executors", (Throwable)e);
        }
        this.executors.put(fullId, executorInfo);
    }

    public ManagedBuffer getBlockData(String appId, String execId, String blockId) {
        String[] blockIdParts = blockId.split("_");
        if (blockIdParts.length < 4) {
            throw new IllegalArgumentException("Unexpected block id format: " + blockId);
        }
        if (!blockIdParts[0].equals("shuffle")) {
            throw new IllegalArgumentException("Expected shuffle block id, got: " + blockId);
        }
        int shuffleId = Integer.parseInt(blockIdParts[1]);
        int mapId = Integer.parseInt(blockIdParts[2]);
        int reduceId = Integer.parseInt(blockIdParts[3]);
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        return this.getSortBasedShuffleBlockData(executor, shuffleId, mapId, reduceId);
    }

    public void applicationRemoved(String appId, boolean cleanupLocalDirs) {
        logger.info("Application {} removed, cleanupLocalDirs = {}", (Object)appId, (Object)cleanupLocalDirs);
        Iterator it = this.executors.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = it.next();
            AppExecId fullId = (AppExecId)entry.getKey();
            final ExecutorShuffleInfo executor = (ExecutorShuffleInfo)entry.getValue();
            if (!appId.equals(fullId.appId)) continue;
            it.remove();
            if (this.db != null) {
                try {
                    this.db.delete(ExternalShuffleBlockResolver.dbAppExecKey(fullId));
                }
                catch (IOException e) {
                    logger.error("Error deleting {} from executor state db", (Object)appId, (Object)e);
                }
            }
            if (!cleanupLocalDirs) continue;
            logger.info("Cleaning up executor {}'s {} local dirs", (Object)fullId, (Object)executor.localDirs.length);
            this.directoryCleaner.execute(new Runnable(){

                @Override
                public void run() {
                    ExternalShuffleBlockResolver.this.deleteExecutorDirs(executor.localDirs);
                }
            });
        }
    }

    private void deleteExecutorDirs(String[] dirs) {
        for (String localDir : dirs) {
            try {
                JavaUtils.deleteRecursively((File)new File(localDir));
                logger.debug("Successfully cleaned up directory: {}", (Object)localDir);
            }
            catch (Exception e) {
                logger.error("Failed to delete directory: " + localDir, (Throwable)e);
            }
        }
    }

    private ManagedBuffer getSortBasedShuffleBlockData(ExecutorShuffleInfo executor, int shuffleId, int mapId, int reduceId) {
        FileSegmentManagedBuffer fileSegmentManagedBuffer;
        block5: {
            File indexFile = ExternalShuffleBlockResolver.getFile(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.index");
            DataInputStream in = null;
            try {
                in = new DataInputStream(new FileInputStream(indexFile));
                in.skipBytes(reduceId * 8);
                long offset = in.readLong();
                long nextOffset = in.readLong();
                fileSegmentManagedBuffer = new FileSegmentManagedBuffer(this.conf, ExternalShuffleBlockResolver.getFile(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.data"), offset, nextOffset - offset);
                if (in == null) break block5;
            }
            catch (IOException e) {
                try {
                    throw new RuntimeException("Failed to open file: " + indexFile, e);
                }
                catch (Throwable throwable) {
                    if (in != null) {
                        JavaUtils.closeQuietly(in);
                    }
                    throw throwable;
                }
            }
            JavaUtils.closeQuietly((Closeable)in);
        }
        return fileSegmentManagedBuffer;
    }

    @VisibleForTesting
    static File getFile(String[] localDirs, int subDirsPerLocalDir, String filename) {
        int hash = JavaUtils.nonNegativeHash((Object)filename);
        String localDir = localDirs[hash % localDirs.length];
        int subDirId = hash / localDirs.length % subDirsPerLocalDir;
        return new File(new File(localDir, String.format("%02x", subDirId)), filename);
    }

    void close() {
        if (this.db != null) {
            try {
                this.db.close();
            }
            catch (IOException e) {
                logger.error("Exception closing leveldb with registered executors", (Throwable)e);
            }
        }
    }

    private static byte[] dbAppExecKey(AppExecId appExecId) throws IOException {
        String appExecJson = mapper.writeValueAsString((Object)appExecId);
        String key = "AppExecShuffleInfo;" + appExecJson;
        return key.getBytes(StandardCharsets.UTF_8);
    }

    private static AppExecId parseDbAppExecKey(String s) throws IOException {
        if (!s.startsWith(APP_KEY_PREFIX)) {
            throw new IllegalArgumentException("expected a string starting with AppExecShuffleInfo");
        }
        String json = s.substring(APP_KEY_PREFIX.length() + 1);
        AppExecId parsed = (AppExecId)mapper.readValue(json, AppExecId.class);
        return parsed;
    }

    @VisibleForTesting
    static ConcurrentMap<AppExecId, ExecutorShuffleInfo> reloadRegisteredExecutors(DB db) throws IOException {
        ConcurrentMap registeredExecutors = Maps.newConcurrentMap();
        if (db != null) {
            Map.Entry e;
            String key;
            DBIterator itr = db.iterator();
            itr.seek(APP_KEY_PREFIX.getBytes(StandardCharsets.UTF_8));
            while (itr.hasNext() && (key = new String((byte[])(e = (Map.Entry)itr.next()).getKey(), StandardCharsets.UTF_8)).startsWith(APP_KEY_PREFIX)) {
                AppExecId id = ExternalShuffleBlockResolver.parseDbAppExecKey(key);
                logger.info("Reloading registered executors: " + id.toString());
                ExecutorShuffleInfo shuffleInfo = (ExecutorShuffleInfo)mapper.readValue((byte[])e.getValue(), ExecutorShuffleInfo.class);
                registeredExecutors.put(id, shuffleInfo);
            }
        }
        return registeredExecutors;
    }

    public static class AppExecId {
        public final String appId;
        public final String execId;

        @JsonCreator
        public AppExecId(@JsonProperty(value="appId") String appId, @JsonProperty(value="execId") String execId) {
            this.appId = appId;
            this.execId = execId;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            AppExecId appExecId = (AppExecId)o;
            return Objects.equal((Object)this.appId, (Object)appExecId.appId) && Objects.equal((Object)this.execId, (Object)appExecId.execId);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.appId, this.execId});
        }

        public String toString() {
            return Objects.toStringHelper((Object)this).add("appId", (Object)this.appId).add("execId", (Object)this.execId).toString();
        }
    }
}

