;;; Tarjan's amortised union-find data structure.
;;; Copyright (c) 1995 by Olin Shivers.

;;; This data structure implements disjoint sets of elements.
;;; Four operations are supported. The implementation is extremely
;;; fast -- any sequence of N operations can be performed in time
;;; so close to linear it's laughable how close it is. See your
;;; intro data structures book for more. The operations are:
;;;
;;; - (base-set nelts) -> set
;;;   Returns a new set, of size NELTS.
;;;
;;; - (set-size s) -> integer
;;;   Returns the number of elements in set S.
;;;
;;; - (union! set1 set2)
;;;   Unions the two sets -- SET1 and SET2 are now considered the same set
;;;   by SET-EQUAL?.
;;;
;;; - (set-equal? set1 set2)
;;;   Returns true <==> the two sets are the same.

;;; Representation: a set is a cons cell. Every set has a "representative"
;;; cons cell, reached by chasing cdr links until we find the cons with
;;; cdr = (). Set equality is determined by comparing representatives using
;;; EQ?. A representative's car contains the number of elements in the set.

;;; The speed of the algorithm comes because when we chase links to find 
;;; representatives, we collapse links by changing all the cells in the path
;;; we followed to point directly to the representative, so that next time
;;; we walk the cdr-chain, we'll go directly to the representative in one hop.


(define (base-set nelts) (cons nelts '()))

;;; Sets are chained together through cdr links. Last guy in the chain
;;; is the root of the set.

(define (get-set-root s)
  (let lp ((r s))			; Find the last pair
    (let ((next (cdr r)))		; in the list. That's
      (cond ((pair? next) (lp next))	; the root r.

	    (else
	     (if (not (eq? r s))	; Now zip down the list again,
		 (let lp ((x s))	; changing everyone's cdr to r.
		   (let ((next (cdr x)))	
		     (cond ((not (eq? r next))
			    (set-cdr! x r)
			    (lp next))))))
	     r)))))			; Then return r.

(define (set-equal? s1 s2) (eq? (get-set-root s1) (get-set-root s2)))

(define (set-size s) (car (get-set-root s)))

(define (union! s1 s2)
  (let* ((r1 (get-set-root s1))
	 (r2 (get-set-root s2))
	 (n1 (set-size r1))
	 (n2 (set-size r2))
	 (n  (+ n1 n2)))

    (cond ((> n1 n2)
	   (set-cdr! r2 r1)
	   (set-car! r1 n))
	  (else
	   (set-cdr! r1 r2)
	   (set-car! r2 n)))))
