;;; Building mazes with union/find disjoint sets.
;;; Copyright (c) 1995 by Olin Shivers.

;;; This is the algorithmic core of the maze constructor.
;;; External dependencies:
;;; - RANDOM-INT
;;; - Union/find code
;;; - bitwise logical functions

(define-record wall
  owner		; Cell that owns this wall.
  neighbor	; The other cell bordering this wall.
  bit)		; Integer -- a bit identifying this wall in OWNER's cell.

(define-record cell
  reachable	; Union/find set -- all reachable cells.
  id		; Identifying info (e.g., the coords of the cell).
  (walls -1)	; A bitset telling which walls are still standing.
  (parent #f)	; For DFS spanning tree construction.
  (mark #f))    ; For marking the solution path.

;;; Iterates in reverse order.

(define (vector-for-each proc v)
  (let lp ((i (- (vector-length v) 1)))
    (cond ((>= i 0)
	   (proc (vector-ref v i))
	   (lp (- i 1))))))


;;; Randomly permute a vector.

(define (permute-vec! v random-state)
  (let lp ((i (- (vector-length v) 1)))
    (cond ((> i 1)
	   (let ((elt-i (vector-ref v i))
		 (j (random-int i random-state)))	; j in [0,i)
	     (vector-set! v i (vector-ref v j))
	     (vector-set! v j elt-i))
	   (lp (- i 1)))))
  v)


;;; This is the core of the algorithm.

(define (dig-maze walls ncells)
  (call-with-current-continuation
    (lambda (quit)
      (vector-for-each
       (lambda (wall)			; For each wall,
	 (let* ((c1   (wall:owner wall)) ; find the cells on
		(set1 (cell:reachable c1))

		(c2   (wall:neighbor wall)) ; each side of the wall
		(set2 (cell:reachable c2)))

	   ;; If there is no path from c1 to c2, knock down the
	   ;; wall and union the two sets of reachable cells.
	   ;; If the new set of reachable cells is the whole set
	   ;; of cells, quit.
	   (if (not (set-equal? set1 set2))
	       (let ((walls (cell:walls c1))	
		     (wall-mask (bitwise-not (wall:bit wall))))
		 (union! set1 set2)
		 (set-cell:walls c1 (bitwise-and walls wall-mask))
		 (if (= (set-size set1) ncells) (quit))))))
       walls))))


;;; Some simple DFS routines useful for determining path length 
;;; through the maze.

;;; Build a DFS tree from ROOT. 
;;; (DO-CHILDREN proc maze node) applies PROC to each of NODE's children.
;;; We assume there are no loops in the maze; if this is incorrect, the
;;; algorithm will diverge.

(define (dfs-maze maze root do-children)
  (let search ((node root) (parent #f))
    (set-cell:parent node parent)
    (do-children (lambda (child)
		   (if (not (eq? child parent))
		       (search child node)))
		 maze node)))

;;; Move the root to NEW-ROOT.

(define (reroot-maze new-root)
  (let lp ((node new-root) (new-parent #f))
    (let ((old-parent (cell:parent node)))
      (set-cell:parent node new-parent)
      (if old-parent (lp old-parent node)))))

;;; How far from CELL to the root?

(define (path-length cell)
  (do ((len 0 (+ len 1))
       (node (cell:parent cell) (cell:parent node)))
      ((not node) len)))

;;; Mark the nodes from NODE back to root. Used to mark the winning path.

(define (mark-path node)
  (let lp ((node node))
    (set-cell:mark node #t)
    (cond ((cell:parent node) => lp))))

