;;; Print out Postscript for a hex maze.
;;; Copyright (c) 1995 by Olin Shivers.

(define (display-line line . maybe-port)
  (apply display line maybe-port)
  (apply newline maybe-port))

;;; Convert a Scheme number to a PostScript numeral.
(define (PSnumeral num)
  (number->string
   (cond ((integer? num)	; Integers are exact.
	  (if (exact? num) num (inexact->exact num)))
	 ((real? num)		; Non-integer reals -> floating point.
	  (if (exact? num) (exact->inexact num) num))
	 (else (error "Cannot be represented as a PostScript numeral" num)))))


(define (draw-hexes ha entry-col draw-line)
  (let* ((ncols (harr:ncols ha))
	 (nrows (harr:nrows ha)))

    ;; Do the top of the top row.
    (let ((y (* 2 (- nrows 1)))) ; y coord of top-left cell at r/c (nrows-1,0).
      (draw-line -2 y  -1 (+ y 1))

      (do ((c 0 (+ 2 c)))
	  ((>= c ncols))
	(let ((x (* c 3)))
	  (if (not (= c entry-col))
	      (draw-line (- x 1) (+ y 1)
			 (+ x 1) (+ y 1)))
	  (let ((c (+ c 1))
		(x (+ x 3))
		(y (+ y 1)))
	    (cond ((< c ncols)
		   (draw-line (- x 2) y  (- x 1) (+ y 1))
		   (if (not (= c entry-col))
		       (draw-line (- x 1) (+ y 1)  (+ x 1) (+ y 1)))
		   (if (< c (- ncols 1))
		       (draw-line (+ x 1) (+ y 1)  (+ x 2) y))))))))

    ;; Do the bottoms of every row.
    (do ((r 0 (+ 1 r)))
	((= r nrows))
      (let ((y (* r 2)))
	(draw-line -2 y -1 (+ y 1))

	(do ((c 0 (+ 1 c)))
	    ((= c ncols))
	  (let ((x (* c 3))
		(y (+ y (bitwise-and c 1)))
		(wall-bits (cell:walls (href/rc ha r c))))
	    (if (not (zero? (bitwise-and wall-bits south-west)))
		(draw-line (- x 2) y  (- x 1) (- y 1)))
	    (if (not (zero? (bitwise-and wall-bits south)))
		(draw-line (- x 1) (- y 1)  (+ x 1) (- y 1)))
	    (if (not (zero? (bitwise-and wall-bits south-east)))
		(draw-line (+ x 1) (- y 1)  (+ x 2) y))))

	(let ((x (* 3 (- ncols 1)))
	      (y (+ y (modulo (- nrows 1) 2))))
	  (draw-line (+ x 1) (+ y 1) (+ x 2) y))))))


;;; HA is the hex-array with the maze; entry at column ENTRY-COL.
;;; EXIT-COL is the exit column. If #f, the solution path will not be shown.
;;; The maze is scaled to be YSIZE x XSIZE inches, 
;;; translated over 0.5 0.5 inches from the lower left corner of the page.
;;; The optional port arg is the destination of the PostScript source.

(define (hexes->ps ha entry-col exit-col xsize ysize . maybe-port)
  (let* ((port (if (null? maybe-port) (current-output-port) (car maybe-port)))
	 (print-line (lambda (line) (display-line line port)))
	 (ncols (harr:ncols ha))
	 (nrows (harr:nrows ha))

	 ;; Print out a line.
	 (draw-line (lambda (x0 y0 x1 y1)
		      (format port "~a ~a m ~a ~a l~%"
			      (PSnumeral x0) (PSnumeral y0)
			      (PSnumeral x1) (PSnumeral y1)))))
			     
    ;; Print out a header.
    (for-each print-line '("%!"
			   "% Hex maze"
			   ""
			   "/m {moveto} def"
			   "/l {lineto} def"
			   ""
			   "gsave"
			   "0 setlinewidth"
			   ""
			   "matrix currentmatrix	% Save the ctm."
			   "36 36 translate		% Shift over half an inch."
			   ))

    ;; Scale the maze.
    (format port "~a ~a scale	% Scale to unit size~%"
	    (PSnumeral (/ (+ 1 (* 3 ncols))))
	    (PSnumeral (/ (+ 1 (* 2 nrows)))))
    (format port "~a ~a scale	% Scale to x=~a y=~a inches~%"
	    (PSnumeral (* xsize 72))
	    (PSnumeral (* ysize 72))
	    (PSnumeral xsize)
	    (PSnumeral ysize))

    (print-line "2 1 translate	% Move origin to 0,0.")
    (newline port)
    
    (draw-hexes ha entry-col draw-line)

    (if exit-col
	(let ((exit-cell (href/rc ha 0 exit-col)))
	  (for-each print-line '("matrix currentmatrix	% swap in the"
				 "exch dup setmatrix	% saved ctm,"
				 "stroke		% stroke,"
				 "exch setmatrix	% back to path ctm."
				 ""
				 ""
				 "% Solution path (in red)"
				 "1 0 0 setrgbcolor"))
	  (let ((xy (cell:id exit-cell)))
	    (format port "~a ~a m~%"
		    (PSnumeral (car xy)) (PSnumeral (cdr xy))))

	  (do ((node (cell:parent exit-cell) (cell:parent node)))
	      ((not node))
	    (let ((xy (cell:id node)))
	      (format port "~a ~a l~%"
		      (PSnumeral (car xy)) (PSnumeral (cdr xy)))))))

    (for-each print-line '("setmatrix		% Restore stroking CTM."
			   "stroke"
			   "grestore"
			   "showpage"))))

;;; This one strokes as it goes, so you don't build up a huge path
;;; for the whole maze.

(define (hexes->incr-ps ha entry-col exit-col xsize ysize . maybe-port)
  (let* ((port (if (null? maybe-port) (current-output-port) (car maybe-port)))
	 (print-line (lambda (line) (display-line line port)))
	 (ncols (harr:ncols ha))
	 (nrows (harr:nrows ha))

	 ;; Print out a line.
	 (draw-line (lambda (x0 y0 x1 y1)
		      (format port "~a ~a m ~a ~a ls~%"
			      (PSnumeral x0) (PSnumeral y0)
			      (PSnumeral x1) (PSnumeral y1)))))
			     
    ;; Print out a header.
    (for-each print-line '("%!"
			   "% Hex maze"
			   ""
			   "/m  {moveto} def"
			   "/ls {lineto stroke} def"
			   "/l  {lineto} def"
			   ""
			   "gsave"
			   "0 setlinewidth"
			   ""
			   "36 36 translate		% Shift over half an inch."
			   ))

    ;; Scale the maze.
    (format port "~a ~a scale	% Scale to unit size~%"
	    (PSnumeral (/ (+ 1 (* 3 ncols))))
	    (PSnumeral (/ (+ 1 (* 2 nrows)))))
    (format port "~a ~a scale	% Scale to x=~a y=~a inches~%"
	    (PSnumeral (* xsize 72))
	    (PSnumeral (* ysize 72))
	    (PSnumeral xsize)
	    (PSnumeral ysize))

    (print-line "2 1 translate	% Move origin to 0,0.")
    (newline port)
    
    (draw-hexes ha entry-col draw-line)

    (if exit-col
	(let ((exit-cell (href/rc ha 0 exit-col)))
	  (for-each print-line '(""
				 "% Solution path (in red)"
				 "1 0 0 setrgbcolor"))
	  (let ((xy (cell:id exit-cell)))
	    (format port "~a ~a m~%"
		    (PSnumeral (car xy)) (PSnumeral (cdr xy))))

	  (do ((node (cell:parent exit-cell) (cell:parent node)))
	      ((not node))
	    (let ((xy (cell:id node)))
	      (format port "~a ~a l~%"
		      (PSnumeral (car xy)) (PSnumeral (cdr xy)))))))

    (for-each print-line '("stroke"
			   "grestore"
			   "showpage"))))

