;;; Hexagonal hackery for maze generation.
;;; Copyright (c) 1995 by Olin Shivers.

;;; External dependencies:
;;; - cell and wall records
;;; - Functional Postscript for HEXES->PATH
;;; - logical functions for bit hacking
;;; - hex array code.

;;; To have the maze span (0,0) to (1,1):
;;; (scale (/ (+ 1 (* 3 ncols))) (/ (+ 1 (* 2 nrows)))
;;;        (translate (point 2 1) maze))

;;; Every elt of the hex array manages his SW, S, and SE wall.
;;; Terminology: - An even column is one whose column index is even. That
;;;                means the first, third, ... columns (indices 0, 2, ...).
;;;              - An odd column is one whose column index is odd. That
;;;                means the second, fourth... columns (indices 1, 3, ...).
;;;              The even/odd flip-flop is confusing; be careful to keep it
;;;              straight. The *even* columns are the low ones. The *odd*
;;;              columns are the high ones.
;;;    _   _
;;;  _/ \_/ \
;;; / \_/ \_/
;;; \_/ \_/ \
;;; / \_/ \_/
;;; \_/ \_/ \
;;; / \_/ \_/
;;; \_/ \_/ \
;;; / \_/ \_/
;;; \_/ \_/
;;;  0 1 2 3

(define (hexes->path ha entry-col)
  (let* ((ncols (harr:ncols ha))
	 (nrows (harr:nrows ha))

	 ;; Accumulate a list of lines.
	 (lines '())
	 (put-line (lambda (x0 y0 x1 y1)
		     (set! lines (cons (line (point x0 y0) (point x1 y1))
				       lines)))))
			  

    ;; Do the top of the top row.
    (let ((y (* 2 (- nrows 1)))) ; y coord of top-left cell at r/c (nrows-1,0).
      (put-line -2 y  -1 (+ y 1))

      (do ((c 0 (+ 2 c)))
	  ((>= c ncols))
	(let ((x (* c 3)))
	  (if (not (= c entry-col))
	      (put-line (- x 1) (+ y 1)
			(+ x 1) (+ y 1)))
	  (let ((c (+ c 1))
		(x (+ x 3))
		(y (+ y 1)))
	    (cond ((< c ncols)
		   (put-line (- x 2) y  (- x 1) (+ y 1))
		   (if (not (= c entry-col))
		       (put-line (- x 1) (+ y 1)  (+ x 1) (+ y 1)))
		   (if (< c (- ncols 1))
		       (put-line (+ x 1) (+ y 1)  (+ x 2) y))))))))

    ;; Do the bottoms of every row.
    (do ((r 0 (+ 1 r)))
	((= r nrows))
      (let ((y (* r 2)))
	(put-line -2 y -1 (+ y 1))

	(do ((c 0 (+ 1 c)))
	    ((= c ncols))
	  (let ((x (* c 3))
		(y (+ y (bitwise-and c 1)))
		(wall-bits (cell:walls (href/rc ha r c))))
	    (if (not (zero? (bitwise-and wall-bits south-west)))
		(put-line (- x 2) y  (- x 1) (- y 1)))
	    (if (not (zero? (bitwise-and wall-bits south)))
		(put-line (- x 1) (- y 1)  (+ x 1) (- y 1)))
	    (if (not (zero? (bitwise-and wall-bits south-east)))
		(put-line (+ x 1) (- y 1)  (+ x 2) y))))

	(let ((x (* 3 (- ncols 1)))
	      (y (+ y (modulo (- nrows 1) 2))))
	  (put-line (+ x 1) (+ y 1) (+ x 2) y))))


    (normalise-path (apply compose lines) nrows ncols)))

(define (solution-path harr exit-col)
  (do ((node (href/rc harr 0 exit-col) (cell:parent node))
       (pts '() (let ((id (cell:id node)))
		  (cons (point (car id) (cdr id)) pts))))
      ((not node)
       (normalise-path (apply line (reverse pts))
		       (harr:nrows harr)
		       (harr:ncols harr)))))

(define (normalise-path path nrows ncols)
    (scale (/ (+ 1 (* 3 ncols))) (/ (+ 1 (* 2 nrows)))
	   (translate (point 2 1) path)))


(define south-west 1)
(define south      2)
(define south-east 4)

(define (gen-maze-array r c)
  (harr-tabulate r c (lambda (x y) (make-cell (base-set 1) (cons x y)))))

;;; This could be made more efficient.
(define (make-wall-vec harr)
  (let* ((nrows (harr:nrows harr))
	 (ncols (harr:ncols harr))
	 (xmax (* 3 (- ncols 1)))

	 ;; Accumulate walls.
	 (walls '())
	 (add-wall (lambda (o n b) ; owner neighbor bit
		     (set! walls (cons (make-wall o n b) walls)))))
	
    ;; Do everything but the bottom row.
    (do ((x (* (- ncols 1) 3) (- x 3)))
	((< x 0))
      (do ((y (+ (* (- nrows 1) 2) (bitwise-and x 1))
	      (- y 2)))
	  ((<= y 1))	; Don't do bottom row.
	  (let ((hex (href harr x y)))
	    (if (not (zero? x))
		(add-wall hex (href harr (- x 3) (- y 1)) south-west))
	    (add-wall hex (href harr x (- y 2)) south)
	    (if (< x xmax)
		(add-wall hex (href harr (+ x 3) (- y 1)) south-east)))))

    ;; Do the SE and SW walls of the odd columns on the bottom row.
    ;; If the rightmost bottom hex lies in an odd column, however,
    ;; don't add it's SE wall -- it's a corner hex, and has no SE neighbor.
    (if (> ncols 1)
	(let ((rmoc-x (+ 3 (* 6 (quotient (- ncols 2) 2)))))
	  ;; Do rightmost odd col.
	  (let ((rmoc-hex (href harr rmoc-x 1)))
	    (if (< rmoc-x xmax) ; Not  a corner -- do E wall.
		(add-wall rmoc-hex (href harr xmax 0) south-east))
	    (add-wall rmoc-hex (href harr (- rmoc-x 3) 0) south-west))

	  (do ((x (- rmoc-x 6) ; Do the rest of the bottom row's odd cols.
		  (- x 6)))
	      ((< x 3))	; 3 is X coord of leftmost odd column.
	    (add-wall (href harr x 1) (href harr (- x 3) 0) south-west)
	    (add-wall (href harr x 1) (href harr (+ x 3) 0) south-east))))

    (list->vector walls)))


;;; Find the cell ctop from the top row, and the cell cbot from the bottom
;;; row such that cbot is furthest from ctop. 
;;; Return [ctop-x, ctop-y, cbot-x, cbot-y].

(define (pick-entrances harr)
  (dfs-maze harr (href/rc harr 0 0) for-each-hex-child)
  (let ((nrows (harr:nrows harr))
	(ncols (harr:ncols harr)))
    (let tp-lp ((max-len -1)
		(entrance #f)
		(exit #f)
		(tcol (- ncols 1)))
      (if (< tcol 0) (values entrance exit)
	  (let ((top-cell (href/rc harr (- nrows 1) tcol)))
	    (reroot-maze top-cell)
	    (receive (max-len entrance exit)
		(let bt-lp ((max-len max-len)
			    (entrance entrance)
			    (exit exit)
			    (bcol (- ncols 1)))
;		  (format #t "~a ~a ~a ~a~%" max-len entrance exit bcol)
		  (if (< bcol 0) (values max-len entrance exit)
		      (let ((this-len (path-length (href/rc harr 0 bcol))))
			(if (> this-len max-len)
			    (bt-lp this-len tcol bcol (- bcol 1))
			    (bt-lp max-len  entrance exit (- bcol 1))))))
	      (tp-lp max-len entrance exit (- tcol 1))))))))
		


;;; Apply PROC to each node reachable from CELL.
(define (for-each-hex-child proc harr cell)
  (let* ((walls (cell:walls cell))
	 (id (cell:id cell))
	 (x (car id))
	 (y (cdr id))
	 (nr (harr:nrows harr))
	 (nc (harr:ncols harr))
	 (maxy (* 2 (- nr 1)))
	 (maxx (* 3 (- nc 1))))
    (if (not (bit-test walls south-west)) (proc (href harr (- x 3) (- y 1))))
    (if (not (bit-test walls south))      (proc (href harr x       (- y 2))))
    (if (not (bit-test walls south-east)) (proc (href harr (+ x 3) (- y 1))))

    ;; NW neighbor, if there is one (we may be in col 1, or top row/odd col)
    (if (and (> x 0)	; Not in first column.
	     (or (<= y maxy)		; Not on top row or
		 (zero? (modulo x 6))))	; not in an odd column.
	(let ((nw (href harr (- x 3) (+ y 1))))
	  (if (not (bit-test (cell:walls nw) south-east)) (proc nw))))

    ;; N neighbor, if there is one (we may be on top row).
    (if (< y maxy)		; Not on top row
	(let ((n (href harr x (+ y 2))))
	  (if (not (bit-test (cell:walls n) south)) (proc n))))

    ;; NE neighbor, if there is one (we may be in last col, or top row/odd col)
    (if (and (< x maxx)	; Not in last column.
	     (or (<= y maxy)		; Not on top row or
		 (zero? (modulo x 6))))	; not in an odd column.
	(let ((ne (href harr (+ x 3) (+ y 1))))
	  (if (not (bit-test (cell:walls ne) south-west)) (proc ne))))))



;;; The top-level
(define (make-maze nrows ncols . maybe-seed)
  (let* ((seed (if (null? maybe-seed) 20 (car maybe-seed)))
	 (cells (gen-maze-array nrows ncols))
	 (walls (permute-vec! (make-wall-vec cells) (random-state seed))))
    (dig-maze walls (* nrows ncols))
    (receive (entrance exit) (pick-entrances cells)
      (let* ((exit-cell (href/rc cells 0 exit))
	     (walls (cell:walls exit-cell)))
	(reroot-maze (href/rc cells (- nrows 1) entrance))
	(mark-path exit-cell)
	(set-cell:walls exit-cell (bitwise-and walls (bitwise-not south)))
	(values cells entrance exit)))))


(define (pmaze nrows ncols)
  (receive (cells entrance exit) (make-maze nrows ncols)
    (print-hexmaze cells entrance)))
