;;; Hex arrays
;;; Copyright (c) 1995 by Olin Shivers.

;;; External dependencies:
;;; - define-record

;;;        ___       ___       ___
;;;       /   \     /   \     /   \
;;;   ___/  A  \___/  A  \___/  A  \___
;;;  /   \     /   \     /   \     /   \
;;; /  A  \___/  A  \___/  A  \___/  A  \
;;; \     /   \     /   \     /   \     /
;;;  \___/     \___/     \___/     \___/
;;;  /   \     /   \     /   \     /   \
;;; /     \___/     \___/     \___/     \
;;; \     /   \     /   \     /   \     /
;;;  \___/     \___/     \___/     \___/
;;;  /   \     /   \     /   \     /   \
;;; /     \___/     \___/     \___/     \
;;; \     /   \     /   \     /   \     /
;;;  \___/     \___/     \___/     \___/

;;; Hex arrays are indexed by the (x,y) coord of the center of the hexagonal
;;; element. Hexes are three wide and two high; e.g., to get from the center
;;; of an elt to its {NW, N, NE} neighbors, add {(-3,1), (0,2), (3,1)}
;;; respectively.
;;;
;;; Hex arrays are represented with a matrix, essentially made by shoving the
;;; odd columns down a half-cell so things line up. The mapping is as follows:
;;;     Center coord      row/column
;;;     ------------      ----------
;;;     (x,  y)        -> (y/2, x/3)
;;;     (3c, 2r + c&1) <- (r,   c)


(define-record harr
  nrows
  ncols
  elts)

(define (harr r c)
  (make-harr r c (make-vector (* r c))))



(define (href ha x y)
  (let ((r (quotient y 2))
	(c (quotient x 3)))
    (vector-ref (harr:elts ha)
		(+ (* (harr:ncols ha) r) c))))

(define (hset! ha x y val)
  (let ((r (quotient y 2))
	(c (quotient x 3)))
    (vector-set! (harr:elts ha)
		 (+ (* (harr:ncols ha) r) c)
		 val)))

(define (href/rc ha r c)
    (vector-ref (harr:elts ha)
		(+ (* (harr:ncols ha) r) c)))

;;; Create a nrows x ncols hex array. The elt centered on coord (x, y)
;;; is the value returned by (PROC x y).

(define (harr-tabulate nrows ncols proc)
  (let ((v (make-vector (* nrows ncols))))

    (do ((r (- nrows 1) (- r 1)))
	((< r 0))
      (do ((c 0 (+ c 1))
	   (i (* r ncols) (+ i 1)))
	  ((= c ncols))
	(vector-set! v i (proc (* 3 c) (+ (* 2 r) (bitwise-and c 1))))))

    (make-harr nrows ncols v)))


(define (harr-for-each proc harr)
  (vector-for-each proc (harr:elts harr)))
