/*
 * Counting the number of nodes reachable from the base
 * @author Richard Jia
 */

includes TopoCtrlMsg;

module CountingNodeM {

  provides interface StdControl;

  uses {
	interface Timer as BaseStationEpochTimer; // only for base station
	interface Timer as ForwardTimer;
	interface Timer as ForwardReplyTimer;
	interface Timer as ReplyTimer;
	interface Timer as RetransTimer;
    interface Leds;
	interface SendMsg;
	interface ReceiveMsg;
	interface StdControl as CommControl;
	interface Random;
	
  }
}

implementation {

  uint16_t intCntReqSeq; // for base station, the sent out; for others, received
  TOS_Msg msg[3]; // 0: request msg (including forwarding) 1: reply msg 2: forwarded reply
  struct TopoCtrlMsg * recvReplyPkt; // latest received packet, for processing
  struct TopoCtrlMsg * recvReqPkt;
  TOS_MsgPtr buffer_reply;
  TOS_MsgPtr buffer_req;
  struct SimpleRoutingTable routeTable[1];
  uint16_t intNodeCount;
  uint16_t intRand;

  command result_t StdControl.init() {
	call Leds.init();
	call CommControl.init();

	atomic {
	  // some local initialization
	  intCntReqSeq = 0;
	  recvReplyPkt = NULL;
	  intNodeCount = 0;
	}

	dbg(DBG_USR2, "%i node init good\n", TOS_LOCAL_ADDRESS);
	return SUCCESS;
  }  

  command result_t StdControl.start() {
	call CommControl.start();

	// only start basestation timer for node 0
	if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
	  call BaseStationEpochTimer.start(TIMER_REPEAT, BASE_STATION_EPOCH_PERIOD);
	}

	return SUCCESS;
  }

  command result_t StdControl.stop() {
	call CommControl.stop();
	call BaseStationEpochTimer.stop();
	call ForwardTimer.stop();
	call ReplyTimer.stop();
	call RetransTimer.stop();

	return SUCCESS;
  }

  // Send out a new counting request 
  //  Only the base station use this
  task void newCountingRequestTask() {
    struct TopoCtrlMsg *pkt;

	if (intCntReqSeq > 0) {
	  dbg(DBG_USR1, "%i Base ### SeqNo=%i FinalCount=%i ###\n", TOS_LOCAL_ADDRESS, intCntReqSeq, intNodeCount);
	}

    atomic {
	  intNodeCount = 0;
      pkt = (struct TopoCtrlMsg *)msg[0].data;
	  pkt->srcNodeID = TOS_LOCAL_ADDRESS;
	  intCntReqSeq++; // first request is no. 1, NOT 0
	  pkt->seqNo = intCntReqSeq;
	  pkt->type = TOPO_CTRL_MSG_DISCOVERY;
    }

    /* Try to send the packet. Note that this will return
     * failure immediately if the packet could not be queued for
     * transmission.
     */
    if (call SendMsg.send(TOS_BCAST_ADDR, sizeof(struct TopoCtrlMsg), &msg[0])) {
	  dbg(DBG_USR1, "%i BCast request %i initiate good\n", TOS_LOCAL_ADDRESS, intCntReqSeq);
    } else {
	  dbg(DBG_USR1, "%i BCast request %i initiate failed\n", TOS_LOCAL_ADDRESS, intCntReqSeq);	  
	}
	
  }

  event result_t BaseStationEpochTimer.fired() {
	
	dbg(DBG_USR3, "%i BaseStationEpochTimer fired\n", TOS_LOCAL_ADDRESS);
	
	if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
	  post newCountingRequestTask();
	}
	
	return SUCCESS;
  }

  // forward request on by broadcast
  task void forwardCoutingRequestTask() {

	if (recvReqPkt != NULL) { // still a valid recv packet
	  struct TopoCtrlMsg *pkt;

	  atomic {
        pkt = (struct TopoCtrlMsg *)msg[0].data;
	    pkt->srcNodeID = TOS_LOCAL_ADDRESS;
  	    pkt->seqNo = intCntReqSeq;
	    pkt->type = TOPO_CTRL_MSG_DISCOVERY;

        /* Try to send the packet. Note that this will return
        * failure immediately if the packet could not be queued for
        * transmission.
        */
	    if (call SendMsg.send(TOS_BCAST_ADDR, sizeof(struct TopoCtrlMsg), &msg[0])) {
	      dbg(DBG_USR1, "%i BCast request %i forward good\n", TOS_LOCAL_ADDRESS, intCntReqSeq);
        } else {
	      dbg(DBG_USR1, "%i BCast request %i forward failed\n", TOS_LOCAL_ADDRESS, recvReqPkt->seqNo);	  
	    }
	  }
	} // end if
  }

  event result_t ForwardTimer.fired() {
	post forwardCoutingRequestTask();

	return SUCCESS;
  } 

  // reply to the counting request by unicast
  task void replyCountingRequestTask() {

	struct TopoCtrlMsg *pkt;

	atomic {
      pkt = (struct TopoCtrlMsg *)msg[1].data;
	  pkt->srcNodeID = TOS_LOCAL_ADDRESS;
  	  pkt->seqNo = intCntReqSeq;
	  pkt->type = TOPO_CTRL_MSG_DISCOVERY_REPLY;

	  pkt->payLoad_Parent = routeTable[0].next;
	  pkt->payLoad_Child = TOS_LOCAL_ADDRESS;
    

      /* Try to send the packet. Note that this will return
       * failure immediately if the packet could not be queued for
       * transmission.
      */
      if (call SendMsg.send(routeTable[0].next, sizeof(struct TopoCtrlMsg), &msg[1])) {
	    dbg(DBG_USR1, "%i UCast reply [%i %i] good\n", TOS_LOCAL_ADDRESS, TOS_LOCAL_ADDRESS, routeTable[0].next);
      } else {
	    dbg(DBG_USR1, "%i UCast reply [%i %i] failed\n", TOS_LOCAL_ADDRESS, TOS_LOCAL_ADDRESS, routeTable[0].next);
	  }
	}

  }

  event result_t ReplyTimer.fired() {
	post replyCountingRequestTask();

	return SUCCESS;
  }

  task void forwardReplyTask() {

	struct TopoCtrlMsg *pkt;

	atomic {
      pkt = (struct TopoCtrlMsg *)msg[2].data;
	  pkt->srcNodeID = TOS_LOCAL_ADDRESS;
  	  pkt->seqNo = recvReplyPkt->seqNo;
	  pkt->type = recvReplyPkt->type;

	  pkt->payLoad_Parent = recvReplyPkt->payLoad_Parent;
	  pkt->payLoad_Child = recvReplyPkt->payLoad_Child;
    

      /* Try to send the packet. Note that this will return
       * failure immediately if the packet could not be queued for
       * transmission.
      */
      if (call SendMsg.send(routeTable[0].next, sizeof(struct TopoCtrlMsg), &msg[2])) {
	    dbg(DBG_USR1, "%i UCast reply [%i %i] forward good\n", TOS_LOCAL_ADDRESS, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);
      } else {
	    dbg(DBG_USR1, "%i UCast reply [%i %i] forward failed\n", TOS_LOCAL_ADDRESS, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);
	  }
    }
  }

  event result_t ForwardReplyTimer.fired() {
	post forwardReplyTask();

	return SUCCESS;
  }

  // handling retransmission
  task void retransPacketTask() {
	// right now, NO retransmission control
	
  }

  // only if TOPO_RETRANS_MAX is > 0, this is called
  // this should be moved to a separate module, 
  // it does not fit here, but bear it for now
  event result_t RetransTimer.fired() {
	if (TOPO_RETRANS_MAX > 0) {
	  post retransPacketTask();
    }

	return SUCCESS;
  }

  // process received messages, the packet is stored in *recvReplyPkt
  // since we are only receiving one type of message, so there is only one handler
  task void processMsg() {

	dbg(DBG_USR3, "%i start processing packet\n", TOS_LOCAL_ADDRESS);

	if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
	  if (recvReplyPkt->type == TOPO_CTRL_MSG_DISCOVERY_REPLY) {
		// we got a valid reply, since we have a reverse broadcast tree,
		// and we use unicast to reply
		atomic {
		  if (recvReplyPkt->seqNo == intCntReqSeq) {
		    intNodeCount++;
		  }
		
		  if (recvReplyPkt->seqNo == intCntReqSeq) {
		    dbg(DBG_USR1, "%i Base got UCast reply from %i [%i %i]\n", TOS_LOCAL_ADDRESS, recvReplyPkt->srcNodeID, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);
		  }
		  else {
		    dbg(DBG_USR1, "%i Base outdated UCast reply Seq=%i [%i %i]\n", TOS_LOCAL_ADDRESS, recvReplyPkt->seqNo, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);
		  }
		}
	  }
	}
	else if (recvReplyPkt->type == TOPO_CTRL_MSG_DISCOVERY) { // discovery pkt from base station
	  if (recvReplyPkt->seqNo > intCntReqSeq) { // new request
		atomic {
		  dbg(DBG_USR1, "%i received BCast request Type=%i Seq=%i Frm=%i\n", TOS_LOCAL_ADDRESS, recvReplyPkt->type, recvReplyPkt->seqNo, recvReplyPkt->srcNodeID);
		  intCntReqSeq = recvReplyPkt->seqNo; // update seq number to the latest

		  // more importantly, populate & update the routing table
		  // in this case, only one entry (base BASE_STATION_ID, next hop parent)
		  //dbg(DBG_USR1, "%i atomic Table[%i %i]\n ", TOS_LOCAL_ADDRESS, );
		  routeTable[0].dest = BASE_STATION_ID;
		  routeTable[0].next = recvReplyPkt->srcNodeID;		
		

		// **** reply delay timer ****
		//intRand = 100 + 50*TOS_LOCAL_ADDRESS;
		post replyCountingRequestTask();
		//call ReplyTimer.start(TIMER_ONE_SHOT, intRand);

		// call ForwardTimer.start(TIMER_ONE_SHOT, MAX_FORWARD_DELAY);
		// let's not worry about delaying and randomization at this moment,
		// simply forward rightaway
		//post forwardCoutingRequestTask();
		intRand = 250;// + 50*TOS_LOCAL_ADDRESS;
		call ForwardTimer.start(TIMER_ONE_SHOT, intRand);
      }
	  } 
	  else {// outdated request - do nothing
		dbg(DBG_USR3, "%i old request %i\n", TOS_LOCAL_ADDRESS, recvReplyPkt->seqNo);
	  }
	} 
	else if (recvReplyPkt->type == TOPO_CTRL_MSG_DISCOVERY_REPLY) {
	  // to forward to base
      
	  //call ForwardReplyTimer.start(TIMER_ONE_SHOT, MAX_FORWARD_REPLY_DELAY);
	  // let's not worry about delaying and randomization at this moment,
	  // simply forward rightaway
	  dbg(DBG_USR1, "%i got UCast reply from %i [%i %i]\n", TOS_LOCAL_ADDRESS, recvReplyPkt->srcNodeID, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);

	  post forwardReplyTask();
	  
	} 
	else {
	  dbg(DBG_USR1, "%i wrong message type\n", TOS_LOCAL_ADDRESS);
	}
  }

  event result_t SendMsg.sendDone(TOS_MsgPtr sent, result_t success) {
	
	if (success == SUCCESS || TOPO_RETRANS_MAX == 0) {
	  return SUCCESS;
	} else {
	  // Fill in here for retransmission control
	  // right now, NO retransmission control
	  
	  return SUCCESS;
	}
  }

  event TOS_MsgPtr ReceiveMsg.receive(TOS_MsgPtr m) {
	TOS_MsgPtr temp = NULL;
	struct TopoCtrlMsg * pkt;
	// filter out packets by group id
	if (m->group != TOS_AM_GROUP) {
	  return m;
	}

	atomic {
	  pkt = (struct TopoCtrlMsg*) m->data;
	  
	  if (pkt->type == TOPO_CTRL_MSG_DISCOVERY) {
		temp = buffer_req;
	    buffer_req = m;

	    recvReqPkt = (struct TopoCtrlMsg*) buffer_req->data;

		// ******
		if (recvReqPkt->seqNo > intCntReqSeq) { // new request
		  atomic {
		    dbg(DBG_USR1, "%i received BCast request Type=%i Seq=%i Frm=%i\n", TOS_LOCAL_ADDRESS, recvReqPkt->type, recvReqPkt->seqNo, recvReqPkt->srcNodeID);
		    intCntReqSeq = recvReqPkt->seqNo; // update seq number to the latest

		    // more importantly, populate & update the routing table
		    // in this case, only one entry (base BASE_STATION_ID, next hop parent)
		    //dbg(DBG_USR1, "%i atomic Table[%i %i]\n ", TOS_LOCAL_ADDRESS, );
		    routeTable[0].dest = BASE_STATION_ID;
		    routeTable[0].next = recvReqPkt->srcNodeID;		

		    // **** reply delay timer ****
		    //intRand = 100;// + 50*TOS_LOCAL_ADDRESS;
		    post replyCountingRequestTask();
		    //call ReplyTimer.start(TIMER_ONE_SHOT, intRand);

		    // call ForwardTimer.start(TIMER_ONE_SHOT, MAX_FORWARD_DELAY);
		    // let's not worry about delaying and randomization at this moment,
		    // simply forward rightaway
		    //post forwardCoutingRequestTask();
		    intRand = 1000;// + 50*TOS_LOCAL_ADDRESS;
		    call ForwardTimer.start(TIMER_ONE_SHOT, intRand);
          }
	    } 
	    else {// outdated request - do nothing
		  dbg(DBG_USR3, "%i old request %i\n", TOS_LOCAL_ADDRESS, recvReqPkt->seqNo);
	    }
		// ******

	  }
	  else if (pkt->type == TOPO_CTRL_MSG_DISCOVERY_REPLY) {
		temp = buffer_reply;
	    buffer_reply = m;

	    recvReplyPkt = (struct TopoCtrlMsg*) buffer_reply->data;

		// ******

		if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) { // only receive and count.
		  atomic {
		  if (recvReplyPkt->seqNo == intCntReqSeq) {
		    intNodeCount++;
		  }
		
		  if (recvReplyPkt->seqNo == intCntReqSeq) {
		    dbg(DBG_USR1, "%i Base got UCast reply from %i [%i %i]\n", TOS_LOCAL_ADDRESS, recvReplyPkt->srcNodeID, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);
		  }
		  else {
		    dbg(DBG_USR1, "%i Base outdated UCast reply Seq=%i [%i %i]\n", TOS_LOCAL_ADDRESS, recvReplyPkt->seqNo, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);
		  }
		  }
		}
		else { // need to forward
		  //call ForwardReplyTimer.start(TIMER_ONE_SHOT, MAX_FORWARD_REPLY_DELAY);
		  // let's not worry about delaying and randomization at this moment,
		  // simply forward rightaway
	  	  dbg(DBG_USR1, "%i got UCast reply from %i [%i %i]\n", TOS_LOCAL_ADDRESS, recvReplyPkt->srcNodeID, recvReplyPkt->payLoad_Child, recvReplyPkt->payLoad_Parent);

	  	  post forwardReplyTask();
		}
		

		// ******
	  }

    }
	return temp;
  }



}
