 /**
 * DaCounter.nc
 * @author Daria Antonova
 */

module DaCounter {
  provides {
    interface StdControl;
  }
  uses {
    interface Timer;
    interface Random;
    interface IntOutput;
    interface SendMsg as Send;
    interface ReceiveMsg as ReceiveCntMsg;
  }
}
implementation {

  //==================================================================
  // Global variables and helper functions
  //==================================================================
  int i;			// counter, temporary variable
  int parent; 		// ID of parent of this node
  int state; 		// count of number of nodes in the subtree
  int randtime; 	// for random timeout for REPLY and DISCOVERY
  int dist_to_base;	// number of hops to base station
  int retransmissions; // the number of times REPLY was sent to
  					   // parent in this epoch
  TOS_Msg tos_msg; 	// for sending/receiving messages
  uint8_t epoch;    // seqno of DISCOVERY packet sent last
  uint8_t child_counts[MAX_NODES]; 	// child subtree counts
  uint8_t child_seqnos[MAX_NODES];	// child last seqnos


  /**
  * This function constructs
  * a CntMsg packet and sends it
  *
  * @param to - destination address
  * @param s - source of the message (TOS_LOCAL_ADDRESS)
  * @param t - type of message DISCOVERY/REPLY
  * @param n - sequence number of the epoch
  * @param c - in DISCOVERY - number of hops to base station
  *			 - in REPLY - number of nodes in the subtree
  *				(including the root of the subtree)
  */
  result_t send_CntMsg(uint16_t to, uint16_t s, uint8_t t, uint8_t n, uint8_t c ) {
  	//TOS_Msg data; // Declaring it here would case non-receiving messages...
  	CntMsg *message = (CntMsg*)tos_msg.data;
  	message->src = s;
  	message->type = t;
  	message->seqno = n;
  	message->subtree_hop_count = c;
  	dbg(DBG_TEMP, "send_CntMsg: sending to %i, s=%i, t=%i, n=%i, c=%i\n", to,s,t,n,c);
	return call Send.send(to, sizeof(CntMsg), &tos_msg);
  }

  /**
  * This function compares two sequence numbers
  * and returns true if the first one is greater
  * than the second (modulo 128, since the numbers
  * are expected to be unsigned 8-bit integers)
  * Mainly this function takes care of wrap around
  * by returning a>b if a is 0.
  */
  bool greater(uint8_t a, uint8_t b) {
	// allow wrap around
	if (a == 0)
		return 1;
	return (a-b)>0;
  }

  /**
  * Calculate the number of motes in the subtree
  *
  * The node keeps an array of counts in children subtrees
  * as well most recent sequence number of REPLY message
  * from the child (index of the array = address of the sensor)
  *
  * If the child did not send any REPLY packets during this
  * epoch, its count is not included in the sum for this epoch
  *
  */
  void calc_state() {
  	child_seqnos[TOS_LOCAL_ADDRESS] = epoch; // update self-count
  	// Lock the arrays for sum calculation
  	atomic {
  		state = 0;
  		for (i=0; i<MAX_NODES; i++) {
			//dbg(DBG_TEMP, "calc_state id %i, seqno %i, count %i \n", i, child_seqnos[i], child_counts[i]);
  			if (child_seqnos[i] == epoch) {
  				state += child_counts[i];
  			}
		}
	} // end atomic
  }

  //==================================================================
  // Main functions
  //==================================================================
  command result_t StdControl.init()
  {
    call Random.init();
    parent = -1; // no parent initially
    state = 1; 	 // count the root of the subtree
	child_counts[TOS_LOCAL_ADDRESS] = 1;
    randtime = 0;
    epoch = 0;
    dist_to_base = 0;
    return SUCCESS;
  }

  command result_t StdControl.start()
  {
	  dbg(DBG_TEMP, "StdControl.start: booting............\n");

    // BASE_STATION: sends first DISCOVERY PACKET and starts epoch 1
	if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
         dbg(DBG_TEMP, "StdControl.start: broadcasting DISCOVERY\n");
         epoch++;
	     send_CntMsg(TOS_BCAST_ADDR, TOS_LOCAL_ADDRESS, DISCOVERY_PACKET, epoch, 0);
	     call Timer.start(TIMER_ONE_SHOT, DISCOVERY_EPOCH_GAP);
	}
	return SUCCESS;
  }

  command result_t StdControl.stop()
  {
    return SUCCESS;
  }

  //==================================================================
  // Timer interrupt handler
  //==================================================================
  /**
  * This function handles two three cases of timer interrupts.
  * The first case is when the base station transmits discovery
  * packet to start the new epoch
  *
  * The second case is when the non-base-station node rebroadcasts
  * DISCOVERY packet. After the initial reception the node waits
  * a small random amount of time and then does the broadcast. This
  * is to avoid collisions with the broadcasts of the neighboring nodes.
  *
  * The third case non-base-station nodes transmitting REPLY packets.
  * REPLY packets with current count are transmitted every
  * REPLY_GAP/dist_to_base plus a small random number.
  * This results in nodes that are farther away from the
  * base station to reply more quickly, resulting in
  * the upstream nodes being able to report the correct
  * counts in the subtree.
  * Thus the base station has to only wait REPLY_GAP milliseconds
  * before it receives the count of nodes in the network.
  *
  * Note: if there are nodes that are very far from the base
  * station they might not be able to respond in time, because
  * the difference between their reply and their parent's reply
  * would be too small. This can be remedies by base station
  * varying REPLY_GAP from time to time, to determine whether
  * increasing the gap results in additional nodes being discovered.
  *
  */
  event result_t Timer.fired()
  {
    dbg(DBG_TEMP, "Timer.fired!!!\n");

	// First broadcast DISCOVERY packet, then schedule reply to parent
	if (retransmissions < 0 && TOS_LOCAL_ADDRESS != BASE_STATION_ID) {
		 retransmissions++;
         dbg(DBG_TEMP, "Timer.fired: forwarding DISCOVERY packet, dist_to_base = %i\n", dist_to_base);
         send_CntMsg(TOS_BCAST_ADDR, TOS_LOCAL_ADDRESS, DISCOVERY_PACKET, epoch, dist_to_base);

	     // Schedule REPLY packet to parent
         randtime = call Random.rand();
         dbg(DBG_TEMP, "Tiner.fired: scheduling timeout in %i\n", REPLY_GAP/dist_to_base+randtime%RAND_TIME_INTERVAL);
         call Timer.start(TIMER_ONE_SHOT, REPLY_GAP/dist_to_base+randtime%RAND_TIME_INTERVAL);
         return SUCCESS;
	}

	// Calculate the current number of nodes in the subtree
	calc_state();
	call IntOutput.output(state);

	// The Base station starts a new epoch
	if (TOS_LOCAL_ADDRESS == BASE_STATION_ID) {
		dbg(DBG_TEMP, "Timer.fired: Base Station: THE FINAL COUNT IN EPOCH %i is %i\n", epoch, state);
		epoch++;
		dbg(DBG_TEMP, "Timer.fired: Base Station is starting new epoch %i\n", epoch);
		send_CntMsg(TOS_BCAST_ADDR, TOS_LOCAL_ADDRESS, DISCOVERY_PACKET, epoch, dist_to_base);
		call Timer.start(TIMER_ONE_SHOT, DISCOVERY_EPOCH_GAP);
	}

	// Non-base-station nodes push REPLY to parent
	// if the number of retransmission has not exceeded
	// MAX_REPTRANSMITIONS and ACK has not been received
	else if (retransmissions < MAX_RETRANSMISSIONS){
		retransmissions++;
	    dbg(DBG_TEMP, "Timer.fired: sending transmission %i of REPLY with state %i\n", retransmissions, state);
		send_CntMsg(parent, TOS_LOCAL_ADDRESS, REPLY_PACKET, epoch, state);
		// Moved code from here to send.done
	}
	else if (retransmissions == MAX_RETRANSMISSIONS) {
		call Timer.stop();
	}


    return SUCCESS;
  }

  //==================================================================
  // Message received event handler
  //==================================================================
  /**
  * This event handler takes care of processing of two
  * kinds of messages - DISCOVERY and REPLY
  *
  * When fresh DISCOVERY message is received, the node
  * schedules a timer for re-broadcasting it once
  *
  * When a REPLY message is received the node updates the
  * corresponding entry in its child_counts and child_seqnos
  * arrays.
  *
  * If the REPLY message received from the child is an old
  * one, they node resends DISCOVERY packet to make the child
  * aware of the current epoch.
  */
  event TOS_MsgPtr ReceiveCntMsg.receive(TOS_MsgPtr m) {
	CntMsg *message = (CntMsg*)m->data;

    dbg(DBG_TEMP, "ReceiveCntMsg: msg %i from node %i\n", (int)message->type, (int)message->src);

    // -----------------------------------------------------------------
    // If this is a first DISCOVERY packet in this epoch - mark parent,
    // forward DISCOVERY packet to neighbors, send reply later
    if (greater((int)message->seqno, epoch) && (int)message->type == DISCOVERY_PACKET && TOS_LOCAL_ADDRESS != BASE_STATION_ID) {

		 // Stop any scheduled replies from previous epoch
		 call Timer.stop();
		 retransmissions = -1;

    	 // Mark parent
    	 epoch = message->seqno;
    	 parent = (int)message->src;
         dbg(DBG_TEMP, "ReceiveCntMsg: set parent %i for node %i\n", parent, TOS_LOCAL_ADDRESS);

         // Forward DISCOVERY packet to neighbors and set dist_to_base
		 dist_to_base = (int)message->subtree_hop_count + 1;
         randtime = call Random.rand();
         dbg(DBG_TEMP, "ReceiveCntMsg: will forward DISCOVERY in %i ms\n", randtime%BROADCAST_TIME_INTERVAL);
         call Timer.start(TIMER_ONE_SHOT, randtime%BROADCAST_TIME_INTERVAL);
    } // end if DISCOVERY packet

    // ----------------------------------------------------------------
    // If this is the REPLY from current epoch,
    // update the count in this child's subtree
    // and output the current number of nodes on display
    else if ((int)message->type == REPLY_PACKET && (int)message->seqno == epoch && (int)message->src < MAX_NODES) {
		i = child_counts[(int)message->src];
        child_counts[(int)message->src] = (int)message->subtree_hop_count;
		// we add new subtree count to the state and subtract the old count
		atomic { state += (int)message->subtree_hop_count - i; }
		// also update the last epoch seqno
		child_seqnos[(int)message->src] = epoch;
	    dbg(DBG_TEMP, "ReceiveCntMsg: display %i\n", state);
		call IntOutput.output(state);
	} // end if REPLY_PACKET from current epoch

	// If we receive reply from child with an old seqno, the child
	// probably did not get the new DISCOVERY packet => resend
	else if ((int)message->type == REPLY_PACKET && (int)message->seqno != epoch) {
	     dbg(DBG_TEMP, "ReceiveCntMsg: resending DISCOVERY packet to %i\n", (int)message->src);
         send_CntMsg((int)message->src, TOS_LOCAL_ADDRESS, DISCOVERY_PACKET, epoch, dist_to_base);
	}

    return m;

  }

  //==================================================================
  // sendDone and ouputComplete event handlers
  //==================================================================
  /**
  * When the sendDone event occurs we check whether the
  * ack bit of TOS_Msg packet was set to determine if we
  * received the ACK from link layer. This seems to work with
  * varying degree of success.
  * If the ACK was not received, we retransmit the REPLY packet
  * at most MAX_RETRANSMISSIONS number of times total.
  */
  event result_t Send.sendDone(TOS_MsgPtr msg, result_t success)
  {
	  // Retransmit more if the ack from link layer is not received
	  if (retransmissions >= 0 && tos_msg.ack == 0 && TOS_LOCAL_ADDRESS != BASE_STATION_ID) {
	  	dbg(DBG_TEMP, "The ack is %i\n", tos_msg.ack);
	  	randtime = call Random.rand();
	  	call Timer.start(TIMER_ONE_SHOT, randtime%RAND_TIME_INTERVAL);
	}
    return SUCCESS;
  }

  event result_t IntOutput.outputComplete(result_t success) {
    return SUCCESS;
  }

}

