trait Stream[A] {

  def hasElement ():Boolean
  def head ():A
  def tail ():Stream[A]

  // Derived operations

  def print ():Unit = {
    if (hasElement()) {
      println("  " + head());
      tail().print()
    } 
  }

  def printN (n:Int):Unit = 
    if (hasElement())
      if (n > 0) {
	println("  " + head())
	tail().printN(n-1)
      } 
      else
	println("  ...")
  
  def sequence (st:Stream[A]):Stream[A] = 
    new Sequence(this,st)

  private class Sequence (st1:Stream[A], st2:Stream[A]) extends Stream[A] {
    def hasElement ():Boolean = {
      st1.hasElement() || st2.hasElement()
    }
    def head ():A = 
      if (st1.hasElement())
	st1.head()
      else
	st2.head()
    def tail ():Stream[A] = 
      if (st1.hasElement())
	new Sequence(st1.tail(),st2)
      else
	st2.tail()
  }

  def zip[B] (st2:Stream[B]):Stream[Pair[A,B]] = 
    new Zip[B](this,st2)

  private 
  class Zip[B] (st1:Stream[A],st2:Stream[B]) extends Stream[Pair[A,B]] {
    def hasElement ():Boolean = {
      st1.hasElement() && st2.hasElement()
    }
    def head ():Pair[A,B] = Pair.create(st1.head(),st2.head())
    def tail ():Stream[Pair[A,B]] = st1.tail().zip(st2.tail())
  }

  def map[B] (f:(A)=>B):Stream[B] = 
    new Map[B](this,f)

  private
  class Map[B] (st:Stream[A], f:(A)=>B) extends Stream[B] {
    def hasElement ():Boolean = st.hasElement()
    def head ():B = f(st.head())
    def tail ():Stream[B] = st.tail().map(f)
  }

  def filter (p:(A)=>Boolean):Stream[A] = 
    new Filter(this,p)

  private 
  class Filter (st:Stream[A], p:(A)=>Boolean) extends Stream[A] {
    def findNext (s:Stream[A]):Stream[A] = 
      if (s.hasElement()) {
	if (p(s.head()))
	  s
	else 
	  findNext(s.tail())
      } else
	s
    def hasElement ():Boolean = findNext(st).hasElement()
    def head ():A = findNext(st).head()
    def tail ():Stream[A] = new Filter(findNext(st).tail(),p)
  }

  def iterator ():Iterator[A] = StreamToIteratorAdapter.create(this)
}







object Stream {

  // EMPTY stream

  def empty[T] ():Stream[T] = 
    new Empty[T]

  private class Empty[T] extends Stream[T] {
    def hasElement ():Boolean = false
    def head ():T = throw new RuntimeException("Stream.empty().head()")
    def tail ():Stream[T] = throw new RuntimeException("Stream.empty().tail()")
  }

  // CONSTANT stream

  def constant[T] (v:T):Stream[T] = 
    new Constant[T](v)

  private class Constant[T] (v:T) extends Stream[T] {
    def hasElement ():Boolean = true
    def head ():T = v
    def tail ():Stream[T] = this
  }

  // INT-FROM stream

  def intsFrom (v:Int):Stream[Int] = 
    new IntsFrom(v)

  private class IntsFrom (v:Int) extends Stream[Int] {
    def hasElement ():Boolean = true
    def head ():Int = v
    def tail ():Stream[Int] = new IntsFrom(v+1)
  }

  def fromIterator[T] (it:Iterator[T]):Stream[T] = 
    IteratorToStreamAdapter.create(it)
}

