trait Stream[A] {

  def hasElement ():Boolean
  def head ():A
  def tail ():Stream[A]
}



object Stream {

  def print[T] (st:Stream[T]):Unit = 
    if (st.hasElement()) {
      println("  " + st.head());
      print(st.tail())
    } 

  def printN[T] (st:Stream[T], n:Int):Unit = 
    if (st.hasElement())
      if (n > 0) {
	println("  " + st.head())
	printN(st.tail(),n-1)
      } 
      else
	println("  ...")

  def sum (st:Stream[Int]):Int = 
    if (st.hasElement())
      st.head()+sum(st.tail())
    else
      0

  // EMPTY stream

  def empty[T] ():Stream[T] = 
    new Empty[T]

  private class Empty[T] extends Stream[T] {

    def hasElement ():Boolean = false

    def head ():T = throw new RuntimeException("Stream.empty().head()")
    def tail ():Stream[T] = throw new RuntimeException("Stream.empty().tail()")
  }


  // CONSTANT stream

  def constant[T] (v:T):Stream[T] = 
    new Constant[T](v)

  private class Constant[T] (v:T) extends Stream[T] {

    def hasElement ():Boolean = true

    def head ():T = v

    def tail ():Stream[T] = this
  }


  // INT-FROM stream

  def intsFrom (v:Int):Stream[Int] = 
    new IntsFrom(v)

  private class IntsFrom (v:Int) extends Stream[Int] {
    
    def hasElement ():Boolean = true

    def head ():Int = v

    def tail ():Stream[Int] = new IntsFrom(v+1)
  }


  // SEQUENCE of streams

  def sequence[T] (st1:Stream[T], st2:Stream[T]):Stream[T] = 
    new Sequence[T](st1,st2)

  private class Sequence[T] (st1:Stream[T], st2:Stream[T]) extends Stream[T] {

    def hasElement ():Boolean = {
      st1.hasElement() || st2.hasElement()
    }

    def head ():T = 
      if (st1.hasElement())
	st1.head()
      else
	st2.head()

    def tail ():Stream[T] = 
      if (st1.hasElement())
	new Sequence[T](st1.tail(),st2)
      else
	st2.tail()
  }



  // ZIP of streams

  def zip[T,U] (st1:Stream[T],st2:Stream[U]):Stream[Pair[T,U]] = 
    new Zip[T,U](st1,st2)

  private 
  class Zip[T,U] (st1:Stream[T],st2:Stream[U]) extends Stream[Pair[T,U]] {

    def hasElement ():Boolean = {
      st1.hasElement() && st2.hasElement()
    }

    def head ():Pair[T,U] = Pair.create(st1.head(),st2.head())

    def tail ():Stream[Pair[T,U]] = new Zip[T,U](st1.tail(),st2.tail())
  }

  // SKIPPING one out of every two elements of a stream

  def oneOutOfTwo[T] (st:Stream[T]):Stream[T] = 
    new OneOutOfTwo[T](st)

  private class OneOutOfTwo[T] (st:Stream[T]) extends Stream[T] {

    def hasElement ():Boolean = st.hasElement()

    def head ():T = st.head()

    def tail ():Stream[T] = 
      if (st.tail().hasElement())
	new OneOutOfTwo(st.tail().tail())
      else
	empty()
  }





  // CARTESIAN PRODUCT of streams

  def cartesian[T,U] (st1:Stream[T],st2:Stream[U]):Stream[Pair[T,U]] = 
    zip(new Duplicates[T](st1,1,1),new Prefixes[U](st2,st2,1,1))

  private 
  class Duplicates[T] (st:Stream[T], init:Int, curr:Int) extends Stream[T] {
    
    def hasElement ():Boolean = st.hasElement()

    def head ():T = st.head()

    def tail ():Stream[T] = 
      if (curr > 1)
	new Duplicates(st,init,curr-1)
      else
	new Duplicates(st.tail(),init+1,init+1)
  }

  private
  class Prefixes[T] (st:Stream[T], ost:Stream[T], 
		     init:Int, curr:Int) extends Stream[T] {
   
    def hasElement ():Boolean = st.hasElement()

    def head ():T = st.head ()

    def tail ():Stream[T] = 
      if (curr > 1)
	new Prefixes(st.tail(),ost,init,curr-1)
      else
	new Prefixes(ost,ost,init+1,init+1)
  }

  
  def map[T,U] (st:Stream[T],f:(T)=>U):Stream[U] = 
    new Map[T,U](st,f)

  private
  class Map[T,U] (st:Stream[T], f:(T)=>U) extends Stream[U] {

    def hasElement ():Boolean = st.hasElement()

    def head ():U = f(st.head())
    
    def tail ():Stream[U] = new Map(st.tail(),f)
  }


  def filter[T] (st:Stream[T],p:(T)=>Boolean):Stream[T] = 
    new Filter[T](st,p)

  private 
  class Filter[T] (st:Stream[T], p:(T)=>Boolean) extends Stream[T] {

    def findNext (s:Stream[T]):Stream[T] = 
      if (s.hasElement()) {
	if (p(s.head()))
	  s
	else 
	  findNext(s.tail())
      } else
	s

    def hasElement ():Boolean = findNext(st).hasElement()

    def head ():T = findNext(st).head()

    def tail ():Stream[T] = new Filter(findNext(st).tail(),p)
  }


  def sieve (st:Stream[Int]):Stream[Int] = 
    new Sieve(st)

  private
  class Sieve (st:Stream[Int]) extends Stream[Int] {

    def hasElement ():Boolean = st.hasElement()

    def head ():Int = st.head()

    def tail ():Stream[Int] = {
      def notDivisibleBy (x:Int):Boolean = !(x % st.head() == 0)
      val multRemoved:Stream[Int] = filter(st.tail(),notDivisibleBy)
      new Sieve(multRemoved)
    }
  }

}
