
object Point {
  
  def cartesian(x:Double,y:Double):Point =
    new CartesianPoint(x,y)
  
  def polar(r:Double,theta:Double):Point = 
    if (r<0) 
      throw new Error("r negative") 
    else 
      new PolarPoint(r,theta)
  
}



abstract class Point {

  def xCoord ():Double
  def yCoord ():Double
  def angleWithXAxis ():Double
  def distanceFromOrigin ():Double
  def distance (q:Point):Double
  def move (dx:Double,dy:Double):Point
  def add (q:Point):Point
  def rotate (theta:Double):Point
  def isEqual (q:Point):Boolean
  def isOrigin ():Boolean
}


class CartesianPoint (xpos:Double, ypos:Double) extends Point {
  
  def xCoord ():Double = 
    xpos
  
  def yCoord ():Double = 
    ypos
  
  def distanceFromOrigin ():Double = 
    math.sqrt(xpos*xpos+ypos*ypos)

  def angleWithXAxis ():Double = 
    math.atan2(ypos,xpos)

  def distance (q:Point):Double = 
    math.sqrt(math.pow(xpos-q.xCoord(),2)+
              math.pow(ypos-q.yCoord(),2))

  def move (dx:Double, dy:Double):Point = 
    new CartesianPoint(xpos+dx,ypos+dy)

  def add (q:Point):Point = 
    this.move(q.xCoord(),q.yCoord())

  def rotate (t:Double):Point = 
    new CartesianPoint(xpos*math.cos(t)-ypos*math.sin(t),
		       xpos*math.sin(t)+ypos*math.cos(t))
  
  def isEqual (q:Point):Boolean = 
    (xpos == q.xCoord()) && (ypos == q.yCoord())

  def isOrigin ():Boolean = 
    (xpos == 0) && (ypos == 0)

  // CANONICAL METHODS

  override def toString ():String = 
    "cartesian(" + xpos + "," + ypos + ")"

  override def equals (other : Any):Boolean = 
    other match {
      case that : Point => this.isEqual(that)
      case _ => false
    }
  
  override def hashCode ():Int = 
    41 * (
      41 + xpos.hashCode()
    ) + ypos.hashCode()

}


class PolarPoint (r:Double, theta:Double) extends Point {

  def xCoord ():Double = r * math.cos(theta)
       
  def yCoord ():Double = r * math.sin(theta)

  def angleWithXAxis ():Double = theta

  def distanceFromOrigin ():Double = r

  def distance (q:Point):Double = 
    math.sqrt(math.pow(xCoord() - q.xCoord(),2) + 
              math.pow(yCoord() - q.yCoord(),2))
  
  def move (dx:Double,dy:Double):Point = 
    new CartesianPoint(xCoord()+dx, yCoord()+dy)

  def add (q:Point):Point = 
    this.move(q.xCoord(), q.yCoord())

  def rotate (angle:Double):Point = 
    new PolarPoint(r, theta+angle)

  private def normalize (angle:Double):Double = 
    if (angle >= 2*math.Pi)
      normalize(angle-2*math.Pi)
    else if (angle < 0)
      normalize(angle+2*math.Pi)
    else
      angle

  def isEqual (q:Point):Boolean =
    (r == q.distanceFromOrigin()) && (normalize(theta) == normalize(q.angleWithXAxis()))

  def isOrigin ():Boolean = (r == 0)

  // CANONICAL METHODS

  override def toString ():String = 
    "polar(" + r + "," + theta + ")"

  override def equals (other : Any):Boolean = 
    other match {
      case that : Point => this.isEqual(that)
      case _ => false
    }
  
  override def hashCode ():Int = 
    41 * 
      (41 + r.hashCode()
    ) + theta.hashCode()
}

