
/* Stream ADT:

   static Stream constant (int)
   static Stream intsFrom (int)
   static Stream cons (int, Stream)
   static Stream sum (Stream, Stream)
   static Stream filter (PredicateFunction, Stream);

   int head ()
   Stream tail ()

   SPECIFICATION:

   constant(i).head() = i
   intsFrom(i).head() = i
   cons(a,S).head() = a
   sum(S1,S2).head() = S1.head() + S2.head()

   constant(i).tail() = constant(i)
   intsFrom(i).tail() = intsFrom(i+1)
   cons(a,S).tail() = S
   sum(S1,S2).tail() = sum(S1.tail(),S2.tail())

*/



public abstract class Stream<A> {

    public static <B> Stream<B> constant (B i) {
        return new StreamConstant<B>(i);
    }

    public static Stream<Integer> intsFrom (int i) {
        return new StreamIntsFrom(i);
    }

    public static <B> Stream<B> cons (B i, Stream<B> s) {
        return new StreamCons<B>(i,s);
    }
        
    public static Stream<Integer> sum (Stream<Integer> s1, Stream<Integer> s2) {
        return new StreamSum(s1,s2);
    }

    public static <B> Stream<B> filter (PredicateFunction<B> pf, Stream<B> s) {
        return new StreamFilter<B>(pf,s);
    }

    static Stream<Integer> sieve (Stream<Integer> s) {
        return new StreamSieve(s);
    }

    public abstract A head ();
    public abstract Stream<A> tail ();
}



class StreamConstant<A> extends Stream<A> {

    private A cnst;
    public StreamConstant (A i) {
        cnst = i;
    }

    public A head () {
        return cnst;
    }

    public Stream<A> tail () {
        return Stream.constant(cnst);
    }

}



class StreamIntsFrom extends Stream<Integer> {

    private int cnst;
    public StreamIntsFrom (int i) {
        cnst = i;
    }

    public Integer head () {
        return cnst;
    }

    public Stream<Integer> tail () {
        return Stream.intsFrom(cnst+1);
    }

}



class StreamCons<A> extends Stream<A> {

    private A first;
    private Stream<A> rest;

    public StreamCons (A i, Stream<A> s) {
        first = i;
        rest = s;
    }

    public A head () {
        return first;
    }

    public Stream<A> tail () {
        return rest;
    }

}



class StreamSum extends Stream<Integer> {

    private Stream<Integer> first;
    private Stream<Integer> second;

    public StreamSum (Stream<Integer> s1, Stream<Integer> s2) {
        first = s1;
        second = s2;
    }

    public Integer head () {
        return first.head() + second.head();
    }

    public Stream<Integer> tail () {
        return Stream.sum(first.tail(),second.tail());
    }

}


class StreamFilter<A> extends Stream<A> {

    private Stream<A> underlying;
    private PredicateFunction<A> predicate;

    public StreamFilter (PredicateFunction<A> pf, Stream<A> s) {
        underlying = s;
        predicate = pf;
    }

    public A head () {
        if (predicate.apply(underlying.head()))
            return underlying.head();
        else
            return Stream.filter(predicate,underlying.tail()).head();
    }

    public Stream<A> tail () {
        if (predicate.apply(underlying.head()))
            return Stream.filter(predicate,underlying.tail());
        else
        return Stream.filter(predicate,underlying.tail()).tail();
    }

}


class StreamSieve extends Stream<Integer> {

    private Stream<Integer> stream;

    public StreamSieve (Stream<Integer> s) {
        stream = s;
    }

    public Integer head () {
        return stream.head();
    }

    public Stream<Integer> tail () {
        Stream<Integer> multRemoved = 
	    Stream.filter(NotDivisibleBy.function(stream.head()),
			  stream.tail());
        return Stream.sieve(multRemoved);
    }
}
