
/*
 * A simple model for the maze
 *
 * A maze is a connected collections of cells
 * where a cell has exits in four directions
 * each direction holding either none() or
 * some(c) for some cell c
 *
 */


public class BasicModel implements Model  {

    private View view;
    private Controller controller;

    /* The current position of the player in the maze */
    private Cell cell;

    /* The current direction faced by the player in the maze */
    private Direction direction;

    protected BasicModel (View v, Controller c) {
	view = v;
	controller = c;
    }

    public static BasicModel create (View v, Controller c) {
	BasicModel m = new BasicModel(v,c);
	m.initializeMaze();
	v.attachToModel(m);
	c.attachToModel(m);
	// initial display of information
	m.updateView();
	return m;
    }


    private static class BasicVisible implements Visible {

	private Direction direction;
	private boolean left,right,front;
	private boolean fleft,fright,ffront;
	private boolean ffleft,ffright,fffront;
	private boolean fwin,ffwin;

	public BasicVisible (Direction d, boolean l, boolean r, boolean f,
			     boolean fl, boolean fr, boolean ff,
			     boolean ffl, boolean ffr, boolean fff,
			     boolean fw, boolean ffw) {
	    direction=d;
	    left=l; right=r;  front=f;
	    fleft=fl; fright=fr; ffront=ff;
	    ffleft=ffl; ffright=ffr; fffront=fff;
	    fwin=fw; ffwin=ffw;
	}

	public Direction direction () { return direction; }
	public boolean left () { return left; }
	public boolean right () { return right; }
	public boolean front () { return front; }
	public boolean frontleft () { return fleft; }
	public boolean frontright () { return fright; }
	public boolean frontfront () { return ffront; }
	public boolean frontfrontleft () { return ffleft; }
	public boolean frontfrontright () { return ffright; }
	public boolean frontfrontfront () { return fffront; }
	public boolean frontwin () { return fwin; }
	public boolean frontfrontwin () { return ffwin; }
    }

    private boolean isOpening (Option<Cell> oc) {
	return (!(oc.isNone()));
    }

    private void updateView () {
	Visible vis;
	boolean l = isOpening(cell.exit(direction.left()));
	boolean r = isOpening(cell.exit(direction.right()));
	Option<Cell> front = cell.exit(direction);
	if (!isOpening(front)) {
	    view.update(new BasicVisible(direction,l,r,false,false,false,false,
					 false,false,false,false,false));
	    return;
	}
	boolean fl = isOpening(front.valOf().exit(direction.left()));
	boolean fr = isOpening(front.valOf().exit(direction.right()));
	Option<Cell> ffront = front.valOf().exit(direction);
	boolean fwin = front.valOf().isWin();
	if (!isOpening(ffront)) {
	    view.update(new BasicVisible(direction,l,r,true,fl,fr,false,
					 false,false,false,fwin,false));
	    return;
	}
	boolean ffl = isOpening(ffront.valOf().exit(direction.left()));
	boolean ffr = isOpening(ffront.valOf().exit(direction.right()));
	Option<Cell> fffront = ffront.valOf().exit(direction);
	boolean ffwin = ffront.valOf().isWin();
	view.update(new BasicVisible(direction,l,r,true,fl,fr,true,
				     ffl,ffr,isOpening(fffront),
				     fwin,ffwin));
    }

    private void initializeMaze () {
	Cell[] cells = new Cell[8];

	// create cells
	for (int i=0; i<8; i++)
	    cells[i] = Cell.create();
	Cell wincell = WinCell.create();

	// connect cells
	cells[0].connect(Direction.SOUTH,cells[1]);
	cells[1].connect(Direction.SOUTH,cells[2]);
	cells[2].connect(Direction.EAST,cells[3]);
	cells[3].connect(Direction.EAST,cells[4]);
	cells[4].connect(Direction.NORTH,cells[5]);
	cells[5].connect(Direction.NORTH,cells[6]);
	cells[6].connect(Direction.WEST,cells[7]);
	cells[7].connect(Direction.WEST,cells[0]);

	cells[4].connect(Direction.SOUTH,wincell);
	
	// choose starting cell + starting direction
	cell = cells[0];
	direction = Direction.NORTH;
    }


    public boolean performLeft () {
	direction=direction.left();
	updateView();
	return true;
    }

    public boolean performRight () {
	direction=direction.right();
	updateView();
	return true;
    }

    public boolean performForward () {
	Option<Cell> res = cell.exit(direction);
	if (res.isNone())
	    return false;
	cell=res.valOf();
	updateView();
	if (cell.isWin()) 
	    view.win();
	return true;
    }

	
    public boolean performBack () {
	Option<Cell> res = cell.exit(direction.opposite());
	if (res.isNone())
	    return false;
	cell=res.valOf();
	updateView();
	return true;
    }

}
